package ai.digital.deploy.gitops.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Consolidated YAML utility that handles scanning for YAML files and reading their content.
 * This utility combines the functionality of YamlFileScanner and prepares YAML content 
 * for processing by GitOpsService but doesn't make API calls itself.
 */
public class YamlHandler {
    
    private static final Logger logger = LoggerFactory.getLogger(YamlHandler.class);
    
    /**
     * Result of YAML file discovery and content reading.
     */
    public static class YamlContent {
        private final String filePath;
        private final String content;
        private final boolean success;
        private final String errorMessage;
        
        public YamlContent(String filePath, String content, boolean success, String errorMessage) {
            this.filePath = filePath;
            this.content = content;
            this.success = success;
            this.errorMessage = errorMessage;
        }
        
        public String getFilePath() { return filePath; }
        public String getContent() { return content; }
        public boolean isSuccess() { return success; }
        public String getErrorMessage() { return errorMessage; }
        
        public static YamlContent success(String filePath, String content) {
            return new YamlContent(filePath, content, true, null);
        }
        
        public static YamlContent failure(String filePath, String errorMessage) {
            return new YamlContent(filePath, null, false, errorMessage);
        }
    }
    
    /**
     * Scans repository for YAML files and reads their content.
     * Returns a list of YamlContent objects ready for GitOpsService processing.
     */
    public static List<YamlContent> scanAndReadYamlFiles(String repositoryPath) {
        logger.info("Scanning for YAML files in repository: {}", repositoryPath);
        
        List<YamlContent> yamlContents = new ArrayList<>();
        
        try {
            // Step 1: Scan for YAML files
            List<String> yamlFiles = scanForYamlFiles(repositoryPath);
            logger.info("Found {} YAML files to process", yamlFiles.size());
            
            // Step 2: Read content from each YAML file
            for (String yamlFilePath : yamlFiles) {
                YamlContent yamlContent = readYamlFile(yamlFilePath);
                yamlContents.add(yamlContent);
                
                if (yamlContent.isSuccess()) {
                    logger.debug("Successfully read YAML file: {}", yamlFilePath);
                } else {
                    logger.warn("Failed to read YAML file: {} - {}", yamlFilePath, yamlContent.getErrorMessage());
                }
            }
            
        } catch (Exception e) {
            logger.error("Error during YAML scanning and reading: {}", e.getMessage(), e);
            // Add an error entry to indicate the overall failure
            yamlContents.add(YamlContent.failure(repositoryPath, "Failed to scan repository: " + e.getMessage()));
        }
        
        long successCount = yamlContents.stream().filter(YamlContent::isSuccess).count();
        logger.info("YAML scanning complete: {} successful, {} failed out of {} total files",
                   successCount, yamlContents.size() - successCount, yamlContents.size());
        
        return yamlContents;
    }
    
    /**
     * Reads content from a single YAML file.
     */
    private static YamlContent readYamlFile(String yamlFilePath) {
        try {
            String content = Files.readString(Paths.get(yamlFilePath));
            
            if (content == null || content.trim().isEmpty()) {
                logger.debug("YAML file is empty: {}", yamlFilePath);
                return YamlContent.success(yamlFilePath, ""); // Empty content is still success
            }
            
            return YamlContent.success(yamlFilePath, content);
            
        } catch (IOException e) {
            logger.error("Failed to read YAML file: {} - {}", yamlFilePath, e.getMessage());
            return YamlContent.failure(yamlFilePath, "IO Error: " + e.getMessage());
        } catch (Exception e) {
            logger.error("Unexpected error reading YAML file: {} - {}", yamlFilePath, e.getMessage());
            return YamlContent.failure(yamlFilePath, "Unexpected error: " + e.getMessage());
        }
    }
    
    /**
     * Scans the specified directory recursively for YAML files (.yml and .yaml extensions).
     * 
     * @param directoryPath The directory path to scan
     * @return List of absolute file paths for found YAML files
     * @throws IOException If there's an error accessing the directory
     */
    public static List<String> scanForYamlFiles(String directoryPath) throws IOException {
        logger.info("Scanning directory for YAML files: {}", directoryPath);
        
        Path startPath = Paths.get(directoryPath);
        if (!Files.exists(startPath) || !Files.isDirectory(startPath)) {
            logger.warn("Directory does not exist or is not a directory: {}", directoryPath);
            return new ArrayList<>();
        }
        
        List<String> yamlFiles = new ArrayList<>();
        
        try (Stream<Path> paths = Files.walk(startPath)) {
            yamlFiles = paths
                .filter(Files::isRegularFile)
                .filter(path -> {
                    String fileName = path.getFileName().toString().toLowerCase();
                    return fileName.endsWith(".yml") || fileName.endsWith(".yaml");
                })
                .map(Path::toAbsolutePath)
                .map(Path::toString)
                .collect(Collectors.toList());
        } catch (IOException e) {
            logger.error("Error scanning directory for YAML files: {}", directoryPath, e);
            throw e;
        }
        
        logger.info("Found {} YAML files in directory: {}", yamlFiles.size(), directoryPath);
        
        // Log found files for debugging
        if (logger.isDebugEnabled()) {
            yamlFiles.forEach(file -> logger.debug("Found YAML file: {}", file));
        }
        
        return yamlFiles;
    }
    
    /**
     * Gets relative path for display purposes.
     */
    public static String getRelativePathForDisplay(String repositoryPath, String filePath) {
        if (repositoryPath != null && filePath != null && filePath.startsWith(repositoryPath)) {
            return filePath.substring(repositoryPath.length()).replaceFirst("^[/\\\\]", "");
        }
        return filePath;
    }
    
    /**
     * Gets a relative path from a base directory for display purposes.
     * Alternative method signature for compatibility.
     * 
     * @param basePath The base directory path
     * @param filePath The full file path
     * @return Relative path string
     */
    public static String getRelativePath(String basePath, String filePath) {
        try {
            Path base = Paths.get(basePath).toAbsolutePath();
            Path file = Paths.get(filePath).toAbsolutePath();
            return base.relativize(file).toString();
        } catch (Exception e) {
            logger.debug("Could not get relative path for {} from {}", filePath, basePath);
            return filePath;
        }
    }
    
    /**
     * Extracts repository name from Git URL.
     * Examples:
     * - https://github.com/user/repo.git -> repo
     * - git@github.com:user/repo.git -> repo
     * - https://github.com/user/repo -> repo
     */
    public static String extractRepositoryName(String url) {
        try {
            // Remove .git suffix if present
            String cleanUrl = url.endsWith(".git") ? url.substring(0, url.length() - 4) : url;
            
            // Extract the last part after / or :
            String repoName;
            if (cleanUrl.contains("/")) {
                repoName = cleanUrl.substring(cleanUrl.lastIndexOf("/") + 1);
            } else if (cleanUrl.contains(":")) {
                repoName = cleanUrl.substring(cleanUrl.lastIndexOf(":") + 1);
            } else {
                repoName = cleanUrl;
            }
            
            // Sanitize repository name for directory creation
            return repoName.replaceAll("[^a-zA-Z0-9_-]", "_");
            
        } catch (Exception e) {
            logger.warn("Failed to extract repository name from URL: {}", url, e);
            return "unknown-repo";
        }
    }
    
    /**
     * Extracts organization/user name from Git URL.
     * Examples:
     * - https://github.com/spring-projects/spring-petclinic.git -> spring-projects
     * - git@github.com:company/my-app.git -> company
     * - https://gitlab.com/my-org/my-project -> my-org
     */
    public static String extractOrganizationName(String url) {
        try {
            // Handle different Git URL formats
            String cleanUrl = url.toLowerCase().trim();
            
            // Remove protocol and authentication info
            cleanUrl = cleanUrl.replaceFirst("^https?://", "");
            cleanUrl = cleanUrl.replaceFirst("^git@", "");
            cleanUrl = cleanUrl.replaceFirst("^[^@]*@", ""); // Remove username@ part
            
            // For SSH format like git@github.com:org/repo.git
            if (cleanUrl.contains(":") && !cleanUrl.startsWith("http")) {
                String[] parts = cleanUrl.split(":");
                if (parts.length > 1) {
                    String pathPart = parts[1];
                    if (pathPart.contains("/")) {
                        String orgName = pathPart.substring(0, pathPart.indexOf("/"));
                        return sanitizeDirectoryName(orgName);
                    }
                }
            }
            
            // For HTTPS format like https://github.com/org/repo.git
            if (cleanUrl.contains("/")) {
                String[] segments = cleanUrl.split("/");
                // Find the organization segment (usually after domain)
                for (int i = 0; i < segments.length - 1; i++) {
                    String segment = segments[i];
                    // Skip domain parts
                    if (segment.contains(".") || segment.isEmpty()) {
                        continue;
                    }
                    // This should be the organization
                    return sanitizeDirectoryName(segment);
                }
            }
            
            return null;
            
        } catch (Exception e) {
            logger.debug("Failed to extract organization name from URL: {}", url, e);
            return null;
        }
    }
    
    /**
     * Sanitizes a string for use as a directory name.
     */
    public static String sanitizeDirectoryName(String name) {
        if (name == null || name.trim().isEmpty()) {
            return "unknown";
        }
        return name.replaceAll("[^a-zA-Z0-9_-]", "_");
    }
    
    /**
     * Creates organization-prefixed repository name for unique directory naming.
     * This ensures consistency across the GitOps plugin.
     * Format: repoName_orgName (if org exists), otherwise just repoName
     */
    public static String createOrgPrefixedRepositoryName(String url) {
        try {
            String repoName = extractRepositoryName(url);
            String orgName = extractOrganizationName(url);
            
            // If organization name is extracted, prefix it to repository name
            if (orgName != null && !orgName.trim().isEmpty() && !orgName.equals("unknown")) {
                return repoName + "_" + orgName;
            } else {
                return repoName;
            }
            
        } catch (Exception e) {
            logger.warn("Failed to create org-prefixed repository name for URL: {}", url, e);
            return "unknown-repo";
        }
    }
}