package com.xebialabs.deployit.plugins.lock;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Environment;

import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

import static java.lang.Boolean.FALSE;

/**
 * Write all in Java, it is cross-platform Create lock.
 * Manager CI that can list logs and clear all locks (control tasks) Default is to use locking for each host
 * (find hostcontainer), can be turned off with synthetic property
 */
public class DeploymentLockContributor {
    private static final String CONCURRENT_DEPLOYMENTS_ALLOWED_PROPERTY = "allowConcurrentDeployments";

    @Contributor
    public static void addDeploymentLockCheckStep(Deltas deltas, DeploymentPlanningContext ctx) {
        DeployedApplication deployedApplication = ctx.getDeployedApplication();
        Environment environment = deployedApplication.getEnvironment();

        Set<ConfigurationItem> cisToBeLocked = checkIfLockingIsRequired(deployedApplication);
        cisToBeLocked.addAll(getContainersRequiringCheck(deltas));

        if (!cisToBeLocked.isEmpty()) {
            boolean enableLockRetry = environment.getProperty("enableLockRetry");
            int lockRetryInterval = environment.getProperty("lockRetryInterval");
            int lockRetryAttempts = environment.getProperty("lockRetryAttempts");
            boolean lockRetryAsync = environment.hasProperty("lockRetryAsync") && ((boolean) environment.getProperty("lockRetryAsync"));
            ctx.addStep(new AcquireAllLocksStep(cisToBeLocked, enableLockRetry, lockRetryInterval, lockRetryAttempts, lockRetryAsync));
        }
    }

    /**
     * 1. on the DeployedApplication
     * 2. on the Environment
     * 3. on the individual containers
     *
     * @param deployedApplication
     * @return
     */
    private static Set<ConfigurationItem> checkIfLockingIsRequired(DeployedApplication deployedApplication) {
        Set<ConfigurationItem> cisToBeLocked = new HashSet<>();

        Environment environment = deployedApplication.getEnvironment();

        if (shouldLockCI(deployedApplication)) {
            cisToBeLocked.add(deployedApplication);
        }

        if (shouldLockCI(environment)) {
            cisToBeLocked.add(environment);
            boolean lockAllContainersInEnvironment = environment.getProperty("lockAllContainersInEnvironment");
            if (lockAllContainersInEnvironment) {
                cisToBeLocked.addAll(environment.getMembers());
            }
        }

        return cisToBeLocked;
    }

    private static boolean shouldLockCI(ConfigurationItem ci) {
        return ci.hasProperty(CONCURRENT_DEPLOYMENTS_ALLOWED_PROPERTY) &&
                FALSE.equals(ci.getProperty(CONCURRENT_DEPLOYMENTS_ALLOWED_PROPERTY));
    }

    private static Set<Container> getContainersRequiringCheck(Deltas deltas) {
        return deltas
                .getDeltas()
                .stream()
                .map(input -> (input.getOperation() == Operation.DESTROY ?
                        input.getPrevious().getContainer() : input.getDeployed().getContainer()))
                .filter(input -> input.hasProperty(CONCURRENT_DEPLOYMENTS_ALLOWED_PROPERTY) &&
                        FALSE.equals(input.getProperty(CONCURRENT_DEPLOYMENTS_ALLOWED_PROPERTY)))
                .collect(Collectors.toSet());
    }
}
