package ai.digital.deploy.notification.email


import ai.digital.deploy.notification.mail.PatSmtpServer
import ai.digital.deploy.notification.freemarker.FreeMarkerConfigurationHolder
import grizzled.slf4j.Logging
import jakarta.activation.{DataHandler, DataSource}
import org.springframework.stereotype.Component
import jakarta.mail.Part
import jakarta.mail.internet.{MimeBodyPart, MimeMultipart}

import java.io.{InputStream, StringWriter}
import java.util
import scala.jdk.CollectionConverters._

@Component
class MultipartEmailSender extends Logging{
  /**
   * Send a custom multipart email with optional template, inline images, and custom logo.
   *
   * @param mailServer        SMTP server to use for sending the email
   * @param from              Sender email address
   * @param to                List of recipient email addresses
   * @param subject           Email subject
   * @param templateVariables Variables to be used in the email template
   * @param baseTemplatePath  Path to the base FreeMarker template (required)
   * @param contentTemplate   Optional path to the content template (inserted into base template)
   * @param textContent       Optional plain text content for the email
   * @param inlineImages      Sequence of (contentId, imagePath) for inline images
   * @param customLogo        Optional (contentId, logoFile) for a custom logo image
   */
  def sendCustomEmail(
    mailServer: PatSmtpServer,
    from: String,
    to: List[String],
    subject: String,
    templateVariables: Map[String, Any],
    baseTemplatePath: Option[String],
    contentTemplate: Option[String] = None,
    textContent: Option[String] = None,
    inlineImages: Seq[(String, String)] = Seq.empty,
    customLogo: Option[(String, String)] = None
  ): Unit = {
    // Create multipart email for HTML with inline images
    val multipart = new MimeMultipart("related")

    // First part is the HTML content
    val htmlPart = new MimeBodyPart()

    val updatedTemplateVars = templateVariables ++
      contentTemplate.map(template => Map("contentTemplatePath" -> template)).getOrElse(Map()) ++
      textContent.map(content => Map("emailTextContent" -> content)).getOrElse(Map())

    val emailContent = renderEmailTemplate(
      baseTemplatePath.getOrElse(throw new IllegalArgumentException("baseTemplatePath must be provided")),
      updatedTemplateVars
    )
    htmlPart.setContent(emailContent, "text/html; charset=utf-8")
    multipart.addBodyPart(htmlPart)

    // Add inline images
    inlineImages.foreach { case (contentId, imagePath) =>
      addInlineImage(multipart, contentId, imagePath)
    }

    // Add custom logo if defined
    customLogo.foreach { case (contentId, logoFile) =>
      if (logoFile.nonEmpty) {
        addCustomLogoImage(multipart, contentId, logoFile)
      }
    }

    mailServer.sendMultipartMessage(
      subject,
      multipart,
      to.asJava,
      from
    )
  }

  private def renderEmailTemplate(templatePath: String, variables: Map[String, Any]): String = {
    val cfg = FreeMarkerConfigurationHolder.getConfiguration
    val template = cfg.getTemplate(templatePath)

    // Convert Scala Map to Java HashMap for FreeMarker
    val dataModel = new util.HashMap[String, Object]()
    variables.foreach { case (key, value) =>
      dataModel.put(key, value.asInstanceOf[Object])
    }

    val writer = new StringWriter()
    template.process(dataModel, writer)
    writer.toString
  }

  private def addInlineImage(multipart: MimeMultipart, contentId: String, imagePath: String): Unit = {
    val imagePart = new MimeBodyPart()

    // Load the image from resources as a stream
    val classLoader = getClass.getClassLoader
    val inputStream = classLoader.getResourceAsStream(imagePath)

    if (inputStream != null) {
      try {
        // Create the attachment directly from the input stream
        imagePart.setDataHandler(new DataHandler(new DataSource {
          override def getContentType: String = getContentTypeFromPath(imagePath)
          override def getInputStream: InputStream = classLoader.getResourceAsStream(imagePath)
          override def getName: String = new java.io.File(imagePath).getName
          override def getOutputStream: java.io.OutputStream =
            throw new java.io.IOException("Read-only data source")
        }))

        imagePart.setHeader("Content-ID", s"<$contentId>")
        imagePart.setDisposition(Part.INLINE);
        multipart.addBodyPart(imagePart)
      } finally {
        inputStream.close()
      }
    } else {
      logger.debug(s"Warning: Logo Image file not found: $imagePath")
    }
  }

  private def addCustomLogoImage(multipart: MimeMultipart, contentId: String, logoFile: String): Unit = {
    if (logoFile != null && logoFile.nonEmpty && logoFile.startsWith("data:")) {
      try {
        val imagePart = new MimeBodyPart()

        // Parse the data URL format: data:contentType;fileName=name;base64,data
        val parts = logoFile.split(";")
        if (parts.length >= 3 && parts(0).startsWith("data:") && parts(1).startsWith("fileName=") && parts(2).startsWith("base64,")) {
          val contentType = parts(0).substring("data:".length)
          val fileName = parts(1).substring("fileName=".length)
          val base64Data = parts(2).substring("base64,".length)

          imagePart.setDataHandler(new DataHandler(new DataSource {
            override def getContentType: String = contentType
            override def getInputStream: InputStream = {
              new java.io.ByteArrayInputStream(java.util.Base64.getDecoder.decode(base64Data.getBytes(java.nio.charset.StandardCharsets.UTF_8)))
            }
            override def getName: String = fileName
            override def getOutputStream: java.io.OutputStream =
              throw new java.io.IOException("Read-only data source")
          }))
          // Set Content-ID header with proper formatting
          imagePart.setHeader("Content-ID", s"<$contentId>")
          imagePart.setDisposition(Part.INLINE);

          multipart.addBodyPart(imagePart)
        }
      } catch {
        case e: Exception =>
          logger.error(s"Failed to process custom logo: ${e.getMessage}", e)
      }
    }
  }

  private def getContentTypeFromPath(imagePath: String): String = {
    val extensionPattern = ".*\\.(\\w+)$".r
    imagePath.toLowerCase match {
      case extensionPattern("png") => "image/png"
      case extensionPattern("jpg") | extensionPattern("jpeg") => "image/jpeg"
      case extensionPattern("gif") => "image/gif"
      case extensionPattern("svg") => "image/svg+xml"
      case _ => "application/octet-stream"
    }
  }
}
