/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin

import java.io.{ByteArrayInputStream, ByteArrayOutputStream, InputStream}
import java.util

import com.fasterxml.jackson.databind.{JsonNode, ObjectMapper}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.ENVIRONMENTS
import com.xebialabs.deployit.plugin.api.udm.Property.Size
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.json_patch.{JsonActivators, JsonPatch, JsonSupport, YamlPatch}
import grizzled.slf4j.Logging
import org.apache.commons.io.IOUtils

/**
 * A PatchDictionary is a CI that contains {@link Environment}-specific entries for non-intrusive dictionary value resolution.
 */
@Metadata(root = ENVIRONMENTS, description = "A JSON Patch Dictionary contains Environment-specific entries for non-intrusive dictionary value resolution")
class JsonPatchDictionary extends BasePatchDictionary with Logging with JsonSupport {
  private final val selectSampleCategory = "Select sample"

  @Property(description = "The patch entries that have been created", size = Size.LARGE, required = true)
  var patches: String = new String

  @Property(description = "The JSON activation entries as JSON Array", size = Size.LARGE, required = false, category = "activators")
  var activators: String = new String

  @Property(
    description = "Regular expression to identify which types of files the patch entries will be applied to",
    defaultValue = ".+\\.(json)",
    hidden = true)
  var jsonFileRegex = new String

  @Property(
    description = "Regular expression to identify which types of files the patch entries will be applied to",
    defaultValue = ".+\\.(yaml|yml)",
    hidden = true)
  var yamlFileRegex = new String

  @Property(
    category = selectSampleCategory,
    description = "The source of the sample that will be provided for patching",
    required = false
  )
  var sourceType = SourceType.Packages

  @Property(
    category = selectSampleCategory,
    description = "List of packages to scan for files containing samples for patches",
    required = false
  )
  var packages: util.Set[String] = new util.HashSet

  @Property(
    category = selectSampleCategory,
    description = "The type of the sample",
    required = false
  )
  var customFileType = SampleFileType.JSON

  @Property(
    category = selectSampleCategory,
    description = "The text of the sample",
    required = false
  )
  var customSample = new String

  @Property(
    description = "Selected artifact for creating patches and/or activators",
    required = false
  )
  var selectedArtifact: String = ""

  @Property(
    description = "Selected YAML document for creating patches and/or activators",
    required = false
  )
  var selectedDocument: String = ""

  case class ApplicationConfig(fileName: String, ci: ConfigurationItem, originalBytes: Array[Byte])

  private def applyIfMatches(kind: String,
                             fileNameRegex: String,
                             objectMapper: ObjectMapper,
                             applyPatch: (String, JsonNode) => InputStream) = new PartialFunction[ApplicationConfig, InputStream] {
    override def isDefinedAt(config: ApplicationConfig): Boolean = config.fileName.matches(fileNameRegex)

    override def apply(config: ApplicationConfig): InputStream = {
      val jsonActivators = new JsonActivators()
      val stream = new ByteArrayInputStream(config.originalBytes)

      logger.debug(s"File [${config.fileName}] recognized as $kind file")
      val documentNode: JsonNode = objectMapper.readTree(stream)
      if (jsonActivators.checkActivators(objectMapper, documentNode, activators)) {
        logger.debug(s"Activations matched for ci [${config.ci}] so proceeding with applying")
        applyPatch(patches, documentNode)
      } else {
        logger.debug(s"Activations not matched for ci [${config.ci}]. Not applying.")
        new ByteArrayInputStream(config.originalBytes)
      }
    }
  }

  override def apply(inputStream: InputStream, ci: ConfigurationItem): InputStream = {
    val jsonPatch = new JsonPatch()
    val yamlPatch = new YamlPatch()

    Option(ci)
      .filter(_.getType.instanceOf(Type.valueOf(classOf[DeployableArtifact])))
      .map { current =>
        val out = new ByteArrayOutputStream()
        IOUtils.copy(inputStream, out)
        val artifact = current.asInstanceOf[DeployableArtifact]
        val filename = artifact.getFile.getName

        applyIfMatches("JSON", jsonFileRegex, jsonMapper, jsonPatch.applyPatchOnJson)
          .orElse(applyIfMatches("YAML", yamlFileRegex, yamlMapper, yamlPatch.applyPatchOnYaml))
          .applyOrElse(
            ApplicationConfig(filename, current, out.toByteArray),
            (_: ApplicationConfig) => throw new Exception("Unsupported file type. Should be JSON or YAML. Is the file extension correct?")
          )
      }.getOrElse {
        logger.debug(s"Patch dictionary can not be applied for ci $ci with type ${Option(ci).map(_.getType).getOrElse("unknown")}")
        inputStream
      }
  }


  def setPatches(patches: String): Unit = this.patches = patches

  def setActivators(activators: String): Unit = this.activators = activators
}

