package com.xebialabs.deployit.report.audit

import java.sql.ResultSet
import java.util
import com.xebialabs.deployit.core.sql.{Pageable, Queries, SchemaInfo}
import com.xebialabs.deployit.engine.api.dto.{AuditPreviewRow, Ordering, Paging}
import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.security.sql.{PermissionsSchema, RolesSchema}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional
import com.xebialabs.deployit.repository.sql.base.{idToPath, pathToId}

import scala.jdk.CollectionConverters._

trait AuditReportRepository {
  def getGlobalAuditReport: util.List[RolePrincipalPermissionRow]

  def getAuditReport(folders: util.List[String]): util.List[AuditPermissionRoleRow]

  def previewAuditReport(folders: util.List[String], ordering: util.List[Ordering],
                         paging: Paging) : AuditPreviewReport
}

@Repository
@Transactional("mainTransactionManager")
class SqlAuditReportRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                              (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends Pageable with AuditReportRepository with AuditReportQuery {

  override def getGlobalAuditReport: util.List[RolePrincipalPermissionRow] =
    jdbcTemplate.query(
      GLOBAL_SELECT,
      new RowMapper[RolePrincipalPermissionRow] {
        override def mapRow(rs: ResultSet, rowNum: Int): RolePrincipalPermissionRow =
          RolePrincipalPermissionRow(
            rs.getString(1),
            rs.getString(2),
            rs.getString(3)
          )
      }
    )

  override def getAuditReport(folders: util.List[String]): util.List[AuditPermissionRoleRow] =
    jdbcTemplate.query(
      LOCAL_SELECT(folders.asScala.toList),
      new RowMapper[AuditPermissionRoleRow] {
        override def mapRow(rs: ResultSet, rowNum: Int): AuditPermissionRoleRow =
          AuditPermissionRoleRow(
            pathToId(rs.getString(1)),
            rs.getString(2),
            rs.getString(3),
            util.UUID.randomUUID().toString
          )
      }
    )

  override def previewAuditReport(folderNames: util.List[String], ordering: util.List[Ordering],
                                  paging: Paging): AuditPreviewReport = {
    val (orderingField, direction) =
      (ordering.asScala.headOption.fold("")(_.field), ordering.asScala.headOption.fold("ASC")(_.direction.toString))
    showPage(paging.page, paging.resultsPerPage)

    val previewQuery: String => util.List[AuditPreviewRow] = (query: String) =>
      jdbcTemplate.query(query, new RowMapper[AuditPreviewRow] {
        override def mapRow(rs: ResultSet, rowNum: Int): AuditPreviewRow =
          AuditPreviewRow(
            rs.getString(1),
            pathToId(rs.getString(2)),
            rs.getString(3),
            rs.getString(4),
            util.UUID.randomUUID().toString
          )
      })

    val previewSelectQuery = selectPreview(folderNames.asScala.toList, orderingField, direction)
    val previewSelectWithPagingQuery = addPaging(previewSelectQuery)
    AuditPreviewReport(
      previewQuery(previewSelectWithPagingQuery),
      previewQuery(previewSelectQuery).size
    )
  }
}

trait AuditReportQuery extends Queries {
  private lazy val GLOBAL_SELECT_FRAGMENT = {
    sqlb"""roleTable.${RolesSchema.Roles.NAME} ROLE_NAME, principalTable.${RolesSchema.RolePrincipals.PRINCIPAL_NAME} PRINCIPAL_NAME, permissionTable.${PermissionsSchema.PERMISSION_NAME} PERMISSION_NAME
          |          from ${RolesSchema.Roles.tableName} roleTable
          |          left join ${PermissionsSchema.tableName} permissionTable on ( permissionTable.${PermissionsSchema.ROLE_ID} = roleTable.${RolesSchema.Roles.ID})
          |          left join ${RolesSchema.RolePrincipals.tableName} principalTable on ( roleTable.${RolesSchema.Roles.ID} = principalTable.${RolesSchema.RolePrincipals.ROLE_ID} )"""
  }

  lazy val GLOBAL_SELECT: String = {
    sqlb"""select $GLOBAL_SELECT_FRAGMENT where permissionTable.${PermissionsSchema.CI_ID} = -1
          |order by roleTable.${RolesSchema.Roles.NAME},
          |principalTable.${RolesSchema.RolePrincipals.PRINCIPAL_NAME},
          |permissionTable.${PermissionsSchema.PERMISSION_NAME}
          |"""
  }

  private lazy val LOCAL_SELECT_FRAGMENT = {
    sqlb"""ciTable.${CIS.path} FOLDER_NAME, permissionTable.${PermissionsSchema.PERMISSION_NAME} PERMISSION_NAME, rolesTable.${RolesSchema.Roles.NAME} ROLE_NAME
          |from ${CIS.tableName} ciTable,
          |${PermissionsSchema.tableName} permissionTable,
          |${RolesSchema.Roles.tableName} rolesTable"""
  }

  private lazy val filterSelectionInCondition: List[String] => String = (folders: List[String]) => {
    if (folders.nonEmpty) sqlb"and ciTable.${CIS.path} in (${folders.map(folder => s"""'${idToPath(folder)}'""").mkString(",")})" else ""
  }

  lazy val LOCAL_SELECT: List[String] => String = (folders: List[String]) => {

    sqlb"""select $LOCAL_SELECT_FRAGMENT where ciTable.${CIS.ci_type} = 'core.Directory'
          |${filterSelectionInCondition(folders)}
          |and ciTable.${CIS.ID} = permissionTable.${PermissionsSchema.CI_ID}
          |and permissionTable.${PermissionsSchema.ROLE_ID} = rolesTable.${RolesSchema.Roles.ID}"""
  }

  private lazy val PREVIEW_SELECT_FRAGMENT = {
    sqlb"""principalTable.${RolesSchema.RolePrincipals.PRINCIPAL_NAME} PRINCIPAL_NAME"""
  }

  private lazy val PREVIEW_FROM_FRAGMENT = {
    sqlb""", ${RolesSchema.RolePrincipals.tableName} principalTable"""
  }

  private lazy val orderByField: String => String = {
    case "folder" => sqlb"ciTable.${CIS.path}"
    case "permission" => sqlb"permissionTable.${PermissionsSchema.PERMISSION_NAME}"
    case "role" => sqlb"rolesTable.${RolesSchema.Roles.NAME}"
    case _ => sqlb"principalTable.${RolesSchema.RolePrincipals.PRINCIPAL_NAME}"
  }

  lazy val selectPreview: (List[String], String, String) => String = (folders: List[String], orderingField: String, direction: String) => {
    sqlb"""select $PREVIEW_SELECT_FRAGMENT, $LOCAL_SELECT_FRAGMENT $PREVIEW_FROM_FRAGMENT where ciTable.${CIS.ci_type} = 'core.Directory'
    ${filterSelectionInCondition(folders)}
    and ciTable.${CIS.ID} = permissionTable.${PermissionsSchema.CI_ID}
    and permissionTable.${PermissionsSchema.ROLE_ID} = rolesTable.${RolesSchema.Roles.ID}
    and principalTable.${RolesSchema.RolePrincipals.ROLE_ID} = rolesTable.${RolesSchema.Roles.ID}
    order by ${orderByField(orderingField)} $direction"""
  }
}

case class RolePrincipalPermissionRow(role: String, principal: String, permission: String)

case class AuditPermissionRoleRow(folder: String, permission: String, role: String, uniqueId: String)
