package com.xebialabs.deployit.report.audit

import ai.digital.deploy.permissions.api.rest.dto.RoleWithPrincipalsDto
import ai.digital.deploy.permissions.client.{GlobalPermissionsServiceClient, PermissionServiceClient, RolePrincipalsServiceClient, RoleServiceClient}
import com.xebialabs.deployit.core.sql.{Pageable, SchemaInfo}
import com.xebialabs.deployit.engine.api.dto.{AuditPreviewRow, Ordering, Paging}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.sql.base.{idToPath, pathToId}
import com.xebialabs.deployit.security.ReadOnlyAdminRoleService
import com.xebialabs.deployit.security.sql.CiResolver
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import java.util
import java.util.UUID
import scala.jdk.CollectionConverters._

trait AuditReportRepository {
  def getGlobalAuditReport: util.List[RolePrincipalPermissionRow]

  def getAuditReport(folders: util.List[String]): util.List[AuditPermissionRoleRow]

  def previewAuditReport(folders: util.List[String], ordering: util.List[Ordering],
                         paging: Paging) : AuditPreviewReport
}

@Repository
@Transactional("mainTransactionManager")
class SqlAuditReportRepository(@Autowired val roleServiceClient: RoleServiceClient,
                               @Autowired val globalPermissionsServiceClient: GlobalPermissionsServiceClient,
                               @Autowired val permissionServiceClient: PermissionServiceClient,
                               @Autowired val rolePrincipalsServiceClient: RolePrincipalsServiceClient,
                               @Autowired val ciResolver: CiResolver)
                              (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends Pageable with AuditReportRepository {

  private def mapPermissionsToRolePrincipalRow(roleName: String,
                                               principal: String,
                                               permissions: List[String]): Seq[RolePrincipalPermissionRow] = {
    permissions.map(
      permission =>
        RolePrincipalPermissionRow(
          roleName,
          principal,
          permission
        )
    )
  }

  private def mapPrincipalsToRolePrincipalsRow(roleWithPrincipals: RoleWithPrincipalsDto,
                                     permissions: List[String]): Seq[RolePrincipalPermissionRow] =
    roleWithPrincipals.principals.flatMap {
      principal =>
        mapPermissionsToRolePrincipalRow(roleWithPrincipals.role.name, principal, permissions)
    }

  override def getGlobalAuditReport: util.List[RolePrincipalPermissionRow] =
    rolePrincipalsServiceClient.readByRolePattern(null)
      .map(roleWithPrincipals =>
        (roleWithPrincipals, globalPermissionsServiceClient.read(roleWithPrincipals.role.name))
      )
      .flatMap {
        case (roleWithPrincipals, permissions) =>
          if (roleWithPrincipals.role.name == ReadOnlyAdminRoleService.ADMIN_READ_ONLY_ROLE_NAME) {
            mapPermissionsToRolePrincipalRow(roleWithPrincipals.role.name, "", permissions)
          }
          else {
            mapPrincipalsToRolePrincipalsRow(roleWithPrincipals, permissions)
          }
      }.sortBy(r => (r.role, r.principal, r.permission)).asJava

  override def getAuditReport(folders: util.List[String]): util.List[AuditPermissionRoleRow] = {
    val roleNames = roleServiceClient.readByRolePattern(null).map(_.name)
    val permissionServiceData = permissionServiceClient.getAllPermissionsForRoles(roleNames).flatMap {
      roleWithPermissions =>
        roleWithPermissions.referencePermissions.flatMap {
          dto =>
            dto.permissions.map(permission => (permission, roleWithPermissions.role.name, dto.reference.toString))

        }
    }
    val references = permissionServiceData.map(_._3).distinct
    val resolvedCis = ciResolver.getResolvedCiFromReferences(references)
      .filter(ci => ci.pk == ci.securedCiPk.get && ci.ciType.instanceOf(Type.valueOf("core.Directory")))

    resolvedCis.flatMap(resolvedCi => {
      permissionServiceData.flatMap {
        case (permission, roleName, reference) => {
          reference match {
            case ref if ref.equals(resolvedCi.securedDirectoryRef.get) =>
              Option(AuditPermissionRoleRow(pathToId(resolvedCi.path), permission, roleName, UUID.randomUUID().toString))
            case _ => None
          }
        }
      }
    }).asJava
  }

  override def previewAuditReport(folderNames: util.List[String], ordering: util.List[Ordering],
                                  paging: Paging): AuditPreviewReport = {
    val (orderingField, direction) =
      (ordering.asScala.headOption.fold("")(_.field), ordering.asScala.headOption.fold("ASC")(_.direction.toString))

    showPage(paging.page, paging.resultsPerPage)
    val permissionServiceData = rolePrincipalsServiceClient.readByRolePattern(null)
      .map(roleWithPrincipals =>
        (roleWithPrincipals, permissionServiceClient.getAllPermissionsForRole(roleWithPrincipals.role.name))
      )
      .flatMap {
        case (roleWithPrincipals, permissions) =>
          roleWithPrincipals.principals.flatMap {
            principal =>
              permissions.referencePermissions.flatMap {
                dto =>
                  dto.permissions.map {
                    permissionName => (principal, permissionName, roleWithPrincipals.role.name, dto.reference.toString)
                  }
              }
          }
      }

    val references = permissionServiceData.map(_._4).distinct
    val resolvedCis = ciResolver.getResolvedCiFromReferences(references).filter(ci => {
      val satisfiesFolderFilter = folderNames match {
        case f if(!f.isEmpty) => folderNames.asScala.toList.map(f => idToPath(f)).contains(ci.path)
        case _ => true
      }
      ci.pk == ci.securedCiPk.get && ci.ciType.instanceOf(Type.valueOf("core.Directory")) && satisfiesFolderFilter
    })

    val auditPreviewRows = resolvedCis.flatMap(resolvedCi => {
      permissionServiceData.flatMap{
        case (principal, permissionName, roleName, reference) =>
          reference match {
            case ref if ref.equals(resolvedCi.securedDirectoryRef.get) =>
              Option(AuditPreviewRow(principal, pathToId(resolvedCi.path), permissionName, roleName, UUID.randomUUID().toString))
            case _ => None
          }
      }
    }).sortBy(row => orderingField match {
      case "principal" => row.principal
      case "role" => row.role
      case "permission" => row.permission
      case "folder" => row.folder
    })

    val auditPreviewReportData = direction match {
      case "DESC" => auditPreviewRows.reverse.asJava
      case _ => auditPreviewRows.asJava
    }

    val(startIndex, endIndex) = resolveAuditReportPage(paging.page, paging.resultsPerPage, auditPreviewReportData.size())

    AuditPreviewReport(
      auditPreviewReportData.subList(startIndex, endIndex),
      auditPreviewReportData.size()
    )
  }

  private def resolveAuditReportPage(page: Int, resultsPerPage: Int, dataSize: Int): (Int, Int) = {
    val startIndex = (page - 1) * resultsPerPage
    val endIndex = (page - 1) * resultsPerPage + resultsPerPage match {
      case e if dataSize < e => dataSize
      case e => e
    }
    (startIndex, endIndex)
  }
}

case class RolePrincipalPermissionRow(role: String, principal: String, permission: String)

case class AuditPermissionRoleRow(folder: String, permission: String, role: String, uniqueId: String)
