package com.xebialabs.deployit.core.sql

class SimpleSelectBuilder(table: TableName)(implicit override val schemaInfo: SchemaInfo) extends AbstractQueryBuilder with Queries {

  private[this] var dist: Boolean = false

  private[this] var count: Boolean = false

  protected[sql] var where: Seq[SqlCondition] = Seq()

  protected[sql] var columns: Seq[(Selectable, Option[String])] = Seq()

  def distinct(): this.type = {
    dist = true
    this
  }

  def select(selectable: Selectable): this.type = {
    columns =  columns :+ (selectable, None)
    this
  }

  def select(selectable: Selectable, alias: String): this.type = {
    columns =  columns :+ (selectable, Some(alias))
    this
  }

  def count(): this.type = {
    count = true
    this
  }

  def where(condition: SqlCondition): this.type = {
    where = where :+ condition
    this
  }

  def parameters: Seq[Any] = {
    where.flatMap(_.arguments)
  }

  protected[sql] def tableExpression: String = table.build(alias)

  protected[sql] def selectFragment: SelectFragment = new SelectFragment(columns.map {
    case (column, None) => column.build(alias)
    case (column, Some(columnAlias)) => schemaInfo.sqlDialect.aliasColumn(column.build(alias), columnAlias)
  }, dist, count)

  protected[sql] def whereFragment: WhereFragment = new WhereFragment(where.map(_.build(alias)))

  override def query: String = sqlb"select $selectFragment from $tableExpression$whereFragment"

}

class SelectBuilder(table: TableName)(implicit override val schemaInfo: SchemaInfo) extends SimpleSelectBuilder(table) {

  private var groupBy: Seq[Selectable] = Seq()

  def groupBy(selectable: Selectable):SelectBuilder = {
    groupBy = groupBy :+ selectable
    this
  }

  def limit(count: Int): SelectBuilder.this.type = showPage(1, count)

  override def query: String =
    addPaging(sqlb"select $selectFragment from $tableExpression$whereFragment$groupByFragment$orderByFragment")

  protected[sql] def groupByFragment: GroupByFragment = new GroupByFragment(groupBy.map(_.build(alias)))
}

class SelectFragmentBuilder(fragment: String)(implicit override val schemaInfo: SchemaInfo) extends SelectBuilder(null) with Queries {
  override def select(selectable: Selectable): this.type = throw new UnsupportedOperationException()

  override def query: String =
    addPaging(sqlb"select $fragment$whereFragment$groupByFragment$orderByFragment")
}

class WrappedJoinBuilder(baseQuery: JoinBuilder)(implicit override val schemaInfo: SchemaInfo) extends SelectBuilder(null) with Queries {

  override def parameters: Seq[Any] = baseQuery.parameters ++ super.parameters

  protected def innerQuery: String =
    sqlb"select ${baseQuery.selectFragment} from ${baseQuery.tableExpression}${baseQuery.whereFragment}${baseQuery.groupByFragment}"

  override def query: String =
    addPaging(sqlb"select $selectFragment from ($innerQuery) $aliasTable $whereFragment$groupByFragment$orderByFragment")

  protected def aliasTable: String = schemaInfo.sqlDialect.aliasTable("", "X")
}

class WrappedCountJoinBuilder(baseQuery: JoinBuilder)(implicit override val schemaInfo: SchemaInfo) extends WrappedJoinBuilder(baseQuery) {
  override def query: String =
    addPaging(sqlb"select count($selectFragment) from ($innerQuery) $aliasTable $whereFragment$groupByFragment$orderByFragment")
}

class DefaultLockSelectBuilder(table: TableName)(implicit override val schemaInfo: SchemaInfo) extends SimpleSelectBuilder(table) {

  override protected[sql] def whereFragment: WhereFragment = new WhereFragment(where.map(_.build(alias)), end = " FOR UPDATE")

}

class MssqlLockSelectBuilder(table: TableName)(implicit override val schemaInfo: SchemaInfo) extends SimpleSelectBuilder(table) {

  override protected[sql] def whereFragment: WhereFragment = new WhereFragment(where.map(_.build(alias)), start = " WITH (UPDLOCK,ROWLOCK) where ")

}
