package com.xebialabs.deployit.core.sql

class JoinBuilder(table: SelectBuilder)(override implicit val schemaInfo: SchemaInfo) extends AbstractQueryBuilder with Queries {

  private[sql] var tables: Seq[SelectBuilder] = Seq(table)

  protected var joinsConditions: Seq[(SqlCondition, JoinType)] = Seq.empty

  def join(table: SelectBuilder, cond: SqlCondition, joinType: JoinType = JoinType.Inner): this.type = {
    tables = tables :+ table
    joinsConditions = joinsConditions :+ (cond, joinType)
    this
  }

  protected[sql] def selectFragment: SelectFragment = tables.zipWithIndex.foldLeft(new SelectFragment(Seq()))(joinOrCombine {
    case (left, right) => left.combine(right.selectFragment)
  })

  protected[sql] def tableExpression: String = {
    val hd :: tl = tables
    (hd.tableExpression +: tl.zip(joinsConditions).map {
      case (t, (c: JoinSubselectCondition, jt)) =>
        s"${jt.fragment} ${schemaInfo.sqlDialect.aliasTable(s"(${t.query})", t.alias.getOrElse(throw new IllegalArgumentException(s"Table expressions ${t.tableExpression} is missing alias.")))} on ${c.build(None)}"
      case (t, (c, jt)) =>
        s"${jt.fragment} ${t.tableExpression} on ${c.build(None)}"
    }).mkString(" ")
  }

  private def joinOrCombine[T, X](combine: (T, X) => T): (T, (X, Int)) => T = {
    case (fragment, (_, index)) if isJoinSubselectCondition(index) => fragment
    case (fragment, (joined, _)) => combine(fragment, joined)
  }

  protected[sql] def whereFragment: WhereFragment = tables.zipWithIndex.foldLeft(new WhereFragment(Seq()))(joinOrCombine {
    case (left, right) => left.combine(right.whereFragment)
  })

  protected[sql] def groupByFragment: GroupByFragment = tables.zipWithIndex.foldLeft(new GroupByFragment(Seq()))(joinOrCombine {
    case (left, right) => left.combine(right.groupByFragment)
  })

  protected[sql] def orderByFragment: OrderByFragment = tables.zipWithIndex.foldLeft(new OrderByFragment(Seq()))(joinOrCombine {
    case (left, right) => left.combine(right.orderByFragment)
  })

  private def isJoinSubselectCondition(index: Int): Boolean = index > 0 && joinsConditions(index - 1)._1.isInstanceOf[JoinSubselectCondition]

  override def query: String = addPaging(sqlb"select $selectFragment from $tableExpression$whereFragment$groupByFragment$orderByFragment")

  override def parameters: Seq[Any] = {
    val hd :: tl = tables
    val (subselects, joins) = tl.zip(joinsConditions).partition {
      case (_, (_: JoinSubselectCondition, _)) => true
      case _ => false
    }
    val subselectTables = subselects.map { case (builder, _) => builder }.iterator
    val joinTables = joins.map { case (builder, _) => builder }

    joinsConditions.flatMap { jc =>
      jc._1 match {
        case cond: JoinSubselectCondition => subselectTables.next().where.flatMap(_.arguments) ++ cond.arguments
        case cond => cond.arguments
      }
    } ++ hd.where.flatMap(_.arguments) ++ joinTables.flatMap(_.where).flatMap(_.arguments)
  }
}

class JoinWithUnionBuilder(union: UnionBuilder)(override implicit val schemaInfo: SchemaInfo) extends JoinBuilder(null) {

  tables = Seq.empty

  protected def unionExpression: String = {
    union.alias.map(a => schemaInfo.sqlDialect.aliasTable("", a)) match {
      case Some(alias) => s"(${union.query}) $alias "
      case None => s"(${union.query})"
    }
  }

  override protected[sql] def tableExpression: String = {
    tables.zip(joinsConditions).map { case (t, (c, jt)) => s"${jt.fragment} ${t.tableExpression} on ${c.build(None)}" }.mkString(" ")
  }

  override protected[sql] def orderByFragment: OrderByFragment =
    tables.foldLeft(new OrderByFragment(union.getOrderBy.map(_.build(union.alias)))) { case (fragment, t) => fragment.combine(t.orderByFragment) }

  override def query: String = addPaging(sqlb"select $selectFragment from $unionExpression$tableExpression$whereFragment$groupByFragment$orderByFragment")

}

sealed abstract class JoinType(val fragment: String)

object JoinType {

  case object Inner extends JoinType("inner join")

  case object Left extends JoinType("left join")

  case object Right extends JoinType("right join")

  case object Full extends JoinType("full join")

}
