package com.xebialabs.deployit.core.upgrade.service

import com.xebialabs.deployit.core.ordering.VersionOrdering

import java.sql.{PreparedStatement, ResultSet}
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.core.upgrade._
import com.xebialabs.deployit.sql.base.schema.CIS.{ID, ci_type, name, parent_id, path, secured_ci, tableName}
import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.sql.base.pathToId
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.matching.Regex

trait LastVersionUpgraderService extends Queries {
  lazy val SELECT_VERSIONS_IN_SYSTEM: String =
    sqlb"SELECT $tableName.$path, $tableName.$name, $tableName.$ID, $secured_ci, $ci_type, $parent_id, ${CI_PROPERTIES.string_value} " +
      sqlb"from ${CI_PROPERTIES.tableName} " +
      sqlb"join $tableName on ${CI_PROPERTIES.ci_id} = $tableName.$parent_id " +
      sqlb"where ${CI_PROPERTIES.tableName}.${CI_PROPERTIES.name} = 'lastVersion' "

  lazy val UPDATE_VERSIONS_QUERY: String = sqlb"UPDATE ${CI_PROPERTIES.tableName} set ${CI_PROPERTIES.string_value} = ? " +
    sqlb"where $name = 'lastVersion' and ${CI_PROPERTIES.ci_id} = ?"

  def fetchCiPropStringValue(): List[ConfigurationItemPropertiesData]

  def findVersionsToUpdate(): List[ConfigurationItemPropertiesData]

  def updateLastVersions(): Unit
}

@Component
@Transactional("mainTransactionManager")
class DefaultLastVersionUpgraderService(
                                         @Autowired @Qualifier("mainSchema") val schemaInfo: SchemaInfo,
                                         @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                         @Autowired @Qualifier("mainTransactionManager") val transactionManager: PlatformTransactionManager,
                                         @Autowired val versionOrdering: VersionOrdering
                                       ) extends LastVersionUpgraderService with Logging {
  override def fetchCiPropStringValue(): List[ConfigurationItemPropertiesData] = {
    jdbcTemplate.query(sqlb"SELECT ${CI_PROPERTIES.ci_id}, ${CI_PROPERTIES.string_value} from ${CI_PROPERTIES.tableName} " +
      sqlb"where $name = 'lastVersion'",
      new RowMapper[ConfigurationItemPropertiesData]() {
        override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationItemPropertiesData = {
          ConfigurationItemPropertiesData(rs.getInt(1), rs.getString(2))
        }
      }).asScala.toList
  }

  override def findVersionsToUpdate(): List[ConfigurationItemPropertiesData] = {
    val cisWithVersions = fetchVersions()
    calculateLastVersion(cisWithVersions)
  }

  override def updateLastVersions(): Unit = {
    val versions = findVersionsToUpdate()
    batchUpdate(UPDATE_VERSIONS_QUERY, versions)
  }

  private def fetchVersions(): List[ConfigurationItemDataVersion] = {
    jdbcTemplate.query(SELECT_VERSIONS_IN_SYSTEM,
      new RowMapper[ConfigurationItemDataVersion]() {
        override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationItemDataVersion = {
          ConfigurationItemDataVersion(
            pathToId(rs.getString(1)),
            rs.getString(2),
            rs.getInt(3),
            rs.getInt(4),
            Type.valueOf(rs.getString(5)),
            rs.getInt(6),
            rs.getString(7)
          )
        }
      }).asScala.toList
  }

  private def calculateLastVersion(apps: List[ConfigurationItemDataVersion]): List[ConfigurationItemPropertiesData] = {
    val lastVersions = mutable.Map[Int, String]()
    apps.foreach { app => {
      val applicationId = app.parentId
      val currentVersion = app.name
      val isNumericVersion = new Regex(s"([0-9.]*)").matches(currentVersion)

      if (VersionOrdering.max(currentVersion, app.version) != app.version & isNumericVersion) {
        lastVersions(applicationId) = lastVersions.get(applicationId) match {
          case Some(lastVersion) =>
            val maxVersion = VersionOrdering.max(lastVersion, currentVersion)
            logger.debug(s"Added version $maxVersion as last version in CI ${app.id.split(app.name).reverse.last}")
            maxVersion
          case None =>
            logger.debug(s"Added version $currentVersion as last version in CI ${app.id.split(app.name).reverse.last}")
            currentVersion
        }
      }
    }
    }
    lastVersions.map(f => ConfigurationItemPropertiesData(f._1, f._2)).toList
  }

  private def batchUpdate(query: String, cis: List[ConfigurationItemPropertiesData]): Unit = {
    jdbcTemplate.batchUpdate(query, new BatchPreparedStatementSetter {
      override def getBatchSize: Int = cis.length

      override def setValues(ps: PreparedStatement, i: Int): Unit = {
        ps.setString(1, cis(i).stringValue)
        ps.setInt(2, cis(i).ciId)
      }
    })
  }
}

