package com.xebialabs.deployit.core.upgrade.service

import com.xebialabs.deployit.core.config.ClientConfiguration
import com.xebialabs.deployit.engine.spi.artifact.resolution.CannotLocateArtifactException
import com.xebialabs.deployit.repository.SearchParameters
import com.xebialabs.deployit.repository.sql.{CiRepository, TypeNotFoundException}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

trait BatchUpgraderService {
  def getBatchCis(query: SearchParameters, batchCallback: SearchParameters => Unit)
}

@Component
class DefaultBatchUpgraderService(@Autowired ciRepository: CiRepository)
  extends BatchUpgraderService with Logging {

  @Autowired
  var clientConfiguration: ClientConfiguration = _

  override def getBatchCis(query: SearchParameters, batchCallback: SearchParameters => Unit): Unit = {
    val batchSize = clientConfiguration.upgradeBatchSize
    val count = ciRepository.count(query)
    logger.info(s"Running batch migration on $count CIs in batches of $batchSize.")
    var pagesIndexed = 0
    val totalPages = (count + batchSize - 1) / batchSize
    while (pagesIndexed < totalPages) {
      val params = query
        .setPage(pagesIndexed)
        .setResultsPerPage(batchSize)
      try {
        batchCallback(params)
      } catch {
        case e: TypeNotFoundException =>
          logger.error(e.getMessage, e.getCause)
          logger.info(s"Skipping page $pagesIndexed due to a missed type.")
        case e: CannotLocateArtifactException =>
          logger.error(e.getMessage, e.getCause)
          logger.info(s"Skipping page $pagesIndexed due to missed artifact.")
        case ex: Exception =>
          throw ex
      } finally {
        pagesIndexed += 1
        logger.info(s"Completed batch $pagesIndexed / total: $totalPages.")
      }
    }
  }
}
