package com.xebialabs.deployit.core.upgrade.service

import java.sql.ResultSet
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.core.upgrade._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.sql.base.pathToId
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._

trait SecurityUpgraderService extends Queries {
  def fetchCis(environmentType: String): List[ConfigurationItemData]

  def fetchOtherCis(): List[ConfigurationItemData]

  def fetchRoles(): List[RoleEntry]

  def fetchRoleRoles(): List[RoleRole]

  def fetchRolePrincipals(): List[RolePrincipal]

  def fetchRolePermissions(): List[PermissionEntry]
}

@Component
@Transactional("mainTransactionManager")
class DefaultSecurityUpgraderService(
                               @Autowired @Qualifier("mainSchema") val schemaInfo: SchemaInfo,
                               @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                               @Autowired @Qualifier("mainTransactionManager") val transactionManager: PlatformTransactionManager
                             ) extends SecurityUpgraderService {


  def fetchCis(ciType: String): List[ConfigurationItemData] = {
    import CISSchema._
    jdbcTemplate.query(sqlb"SELECT $pathColumn, $nameColumn, $idColumn, $securedCiColumn, $ciTypeColumn from $tableName where $ciTypeColumn = ?",
      new RowMapper[ConfigurationItemData]() {
        override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationItemData = {
          ConfigurationItemData(pathToId(rs.getString(1)), rs.getString(2), rs.getInt(3), rs.getInt(4), Type.valueOf(rs.getString(5)))
        }
      }, ciType).asScala.toList
  }

  def fetchOtherCis(): List[ConfigurationItemData] = {
    import CISSchema._
    jdbcTemplate.query(sqlb"SELECT $pathColumn, $nameColumn, $idColumn, $securedCiColumn, $ciTypeColumn from $tableName",
      new RowMapper[ConfigurationItemData]() {
        override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationItemData = {
          ConfigurationItemData(pathToId(rs.getString(1)), rs.getString(2), rs.getInt(3), rs.getInt(4), Type.valueOf(rs.getString(5)))
        }
      }).asScala.toList
  }

  def fetchRoles(): List[RoleEntry] = {
    import RolesSchema.Roles._
    jdbcTemplate.query(sqlb"SELECT $ID, $NAME, $CI_ID from $tableName",
      new RowMapper[RoleEntry]() {
        override def mapRow(rs: ResultSet, rowNum: Int): RoleEntry = {
          RoleEntry(rs.getString(1), rs.getString(2), rs.getInt(3))
        }
      }).asScala.toList
  }

  def fetchRoleRoles(): List[RoleRole] = {
    import RolesSchema.RoleRoles._

    jdbcTemplate.query(sqlb"SELECT $ROLE_ID, $MEMBER_ROLE_ID from $tableName",
      new RowMapper[RoleRole]() {
        override def mapRow(rs: ResultSet, rowNum: Int): RoleRole = {
          RoleRole(rs.getString(1), rs.getString(2))
        }
      }).asScala.toList
  }

  def fetchRolePrincipals(): List[RolePrincipal] = {
    import RolesSchema.RolePrincipals._

    jdbcTemplate.query(sqlb"SELECT $ROLE_ID, $PRINCIPAL_NAME from $tableName",
      new RowMapper[RolePrincipal]() {
        override def mapRow(rs: ResultSet, rowNum: Int): RolePrincipal = {
          RolePrincipal(rs.getString(1), rs.getString(2))
        }
      }).asScala.toList
  }

  def fetchRolePermissions(): List[PermissionEntry] = {
    import PermissionsSchema._

    jdbcTemplate.query(sqlb"SELECT $ROLE_ID, $PERMISSION_NAME, $CI_ID from $tableName",
      new RowMapper[PermissionEntry]() {
        override def mapRow(rs: ResultSet, rowNum: Int): PermissionEntry = {
          PermissionEntry(rs.getString(1), rs.getString(2), rs.getInt(3))
        }
      }).asScala.toList
  }

}
