package com.xebialabs.deployit.core.upgrade

import com.xebialabs.deployit.core.upgrade.service.{ArchiveSecurityUpgraderService, SecurityUpgraderService}
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class Deployit950ReportingImprovementsUpgrader
(
  @Autowired val securityUpgraderService: SecurityUpgraderService,
  @Autowired val archiveSecurityUpgraderService: ArchiveSecurityUpgraderService
) extends Upgrade with Logging {

  private val environmentType = Type.valueOf("udm.Environment")
  private val applicationType = Type.valueOf("udm.Application")

  override def doUpgrade(): Boolean = Try {
    updateInternalAndSecureCiIds()
    updateRolesAndPermissionsSchema()
  } match {
    case Success(_) =>
      logger.debug("Reporting updates have been successfully applied.")
      true
    case Failure(exception: Exception) =>
      logger.error("Reporting updates cannot be applied.", exception)
      false
  }

  override def upgradeVersion(): Version = Version.valueOf("deployit", "9.5.0")

  private def updateInternalAndSecureCiIds(): Unit = {
    logger.info("Updating existing archive database with Security data")
    var environmentTypes: List[Type] = List(environmentType)
    var applicationTypes: List[Type] = List(applicationType)
    val customEnvTypes = DescriptorRegistry.getSubtypes(environmentType).asScala.toList
    val customAppTypes = DescriptorRegistry.getSubtypes(applicationType).asScala.toList

    if (customEnvTypes.nonEmpty) {
      environmentTypes = environmentTypes ++ customEnvTypes
    }
    if (customAppTypes.nonEmpty) {
      applicationTypes = applicationTypes ++ customAppTypes
    }

    environmentTypes.foreach(envType => {
      val cisEnvs = securityUpgraderService.fetchCis(envType.toString)
      archiveSecurityUpgraderService.updateEnvironments(cisEnvs)
    })
    applicationTypes.foreach(appType => {
      val cisApps = securityUpgraderService.fetchCis(appType.toString)
      archiveSecurityUpgraderService.updateApplications(cisApps)
    })

    val otherCis = securityUpgraderService.fetchOtherCis()
    archiveSecurityUpgraderService.updateTargetCis(otherCis)
    logger.info("Finished updating existing archive database with Security data")
  }

  private def updateRolesAndPermissionsSchema(): Unit = {
    val roles = securityUpgraderService.fetchRoles()
    val roleRoles = securityUpgraderService.fetchRoleRoles()
    val rolePrincipals = securityUpgraderService.fetchRolePrincipals()
    val rolePermissions = securityUpgraderService.fetchRolePermissions()

    logger.info("Inserting roles and permission records to the archive database")
    archiveSecurityUpgraderService.insertArchiveRoles(roles)
    archiveSecurityUpgraderService.insertArchiveRoleRoles(roleRoles)
    archiveSecurityUpgraderService.insertArchiveRolePrincipals(rolePrincipals)
    archiveSecurityUpgraderService.insertArchiveRolePermissions(rolePermissions)
    logger.info("Finished inserting  roles and permission records to the archive database")
  }
}


