package com.xebialabs.deployit.permissions.upgrade.service

import ai.digital.deploy.permissions.client.{PermissionServiceClient, RoleServiceClient}
import com.xebialabs.deployit.core.upgrade.PermissionEntry
import com.xebialabs.deployit.core.upgrade.service.SecurityUpgraderService
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.security.sql.CiResolver
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.util.UUID
import scala.util.{Failure, Success, Try}

@Service
class PermissionServiceUpgraderImpl(ciResolver: CiResolver,
                                    securityUpgraderService: SecurityUpgraderService,
                                    roleServiceClient: RoleServiceClient,
                                    permissionServiceClient: PermissionServiceClient
  ) extends PermissionServiceUpgrader with Logging {
  override def isPermissionServiceAttached: Boolean =
    Try {
      roleServiceClient.getAll
    } match {
      case Success(result) => result.nonEmpty
      case Failure(e) => throw e
    }

  private def groupByRoleName(deployRolesMap: Map[String, String], permissionEntries: List[PermissionEntry]): Map[String, List[PermissionEntry]] = {
    permissionEntries.groupBy(_.roleId).map {
      case (id, permissionEntries) =>
        (deployRolesMap.getOrElse(id,
          throw IllegalUpgradeStateException(s"Could not find roleId ${id} in Deploy roles mapping")),
          permissionEntries)
    }
  }

  override def attachPermissionService(): Unit = {
    info(s"Attaching Permission Service - provisioning Roles and Permissions to schema")
    val roles = securityUpgraderService.fetchRoles()
    val rolesPrincipals = securityUpgraderService.fetchRolePrincipals()

    // roles fetch and map
    val rolePrincipalMap =
      rolesPrincipals.map(rolePrincipal => rolePrincipal.roleId -> rolePrincipal.principalName).groupBy(_._1)
        .map { case (id, principalName) => (id, principalName.map(_._2))}

    // roles attach
    roles.map(role =>
      roleServiceClient.create(role.name, rolePrincipalMap.getOrElse(role.id, List[String]()))
    )

    // permissions fetch
    val deployPermissions = securityUpgraderService.fetchRolePermissions()
    val deployRolesMap = roles.map(role => (role.id, role.name)).toMap


    // permissions attach
    groupByRoleName(deployRolesMap, deployPermissions).foreach {
      case (roleName, permissionEntries) =>
        permissionEntries.groupBy(permissionEntry => Option(permissionEntry.ciId)).foreach {
          case (Some(id), permissionEntries) if id != -1 && id != 0 =>
            val referenceId = ciResolver.resolveDirectoryReference(id)
            val permissionNames = permissionEntries.map(_.permissionName)
            permissionServiceClient.createOrUpdate(Option(UUID.fromString(referenceId)), roleName, permissionNames, List())
          case (_, permissionEntries) =>
            val permissionNames = permissionEntries.map(_.permissionName)
            permissionServiceClient.createOrUpdate(None, roleName, permissionNames, List())
        }
    }
    info(s"""Finished attaching Permission Service:
                   * ${roles.size} roles
                   * ${rolesPrincipals.size} principals
                   * ${deployPermissions.size} permissions""")
  }

  override def detachPermissionService(): Unit = {
    info(s"Detaching Permission Service - removing staled Roles and Permissions from Permission Service")
    // remove all the data from permission schema tables
    permissionServiceClient.removeAll()
    roleServiceClient.removeAll()
  }
}

final case class IllegalUpgradeStateException(msg: String) extends DeployitException(msg)
