package com.xebialabs.deployit.core.upgrade.service

import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{JoinBuilder, JoinType, Queries, SchemaInfo, SelectBuilder, SqlCondition, SqlFunction}
import com.xebialabs.deployit.core.upgrade.service.ProfileSchema.{xldProfileUSERNAME, xldProfilelastLoginTime}
import com.xebialabs.deployit.core.upgrade.service.XlUsersSchema.{xlUsersTableAlias, xlUsersUSERNAME, xlUserstableName}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional
import com.xebialabs.deployit.security.sql.SqlXldUserProfileRepository.userProfileTableAlias
import com.xebialabs.deployit.security.sql.XldUserProfileSchema.{ANALYTICS_ENABLED, IS_INTERNAL, LAST_ACTIVE, USERNAME, tableName}
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._
import java.sql.{PreparedStatement, ResultSet, Timestamp}

trait UserProfileUpgradeService extends Queries {

  def fetchExternalUserForInsert(): List[UserProfile]
  def fetchExternalUserForUpdate(): List[UserProfile]
  def fetchInternalUserForUpdate(): List[UserProfile]
  def insertXldUserProfile(userProfileList : List[UserProfile])
  def updateXldUserProfile(userProfileList : List[UserProfile])
  def fetchRecords() : List[UserProfile]
}

@Component
@Transactional("mainTransactionManager")
class DefaultUserProfileUpgradeService (@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                 @Autowired @Qualifier("mainTransactionManager") val transactionManager: PlatformTransactionManager)
                                (@Autowired @Qualifier("mainSchema") override implicit val schemaInfo: SchemaInfo) extends UserProfileUpgradeService with Logging {

  /**
   * This method fetches a list of external user profiles that need to be inserted into the `XLD_USER_PROFILES` table.
   *
   * The method constructs and executes a SQL query to fetch `username` and `last active` entries from the `XLD_PROFILES` table.
   * These entries are those that do not have corresponding entries in either the `XLD_USER_PROFILES` table or the `XL_USERS` table.
   *
   * The result is a list of `UserProfile` objects representing external users that need to be inserted into the `XLD_USER_PROFILES` table.
   *
   * @return A list of `UserProfile` objects representing external users that need to be inserted.
   */
  override def fetchExternalUserForInsert(): List[UserProfile] = {
    import com.xebialabs.deployit.security.sql.XldUserProfileSchema._
    import XlUsersSchema._
    import ProfileSchema._

    val collationAppliedUsername = schemaInfo.sqlDialect.selectWithCollation(xldProfileUSERNAME.build(Some(xldProfileTableAlias)))
    val collationAppliedLowerUsername = schemaInfo.sqlDialect.selectWithCollation(SqlFunction.lower(xldProfileUSERNAME).build(Some(xldProfileTableAlias)))
    val selectBuilder = new SelectBuilder(xldProfileTableName).select(xldProfileUSERNAME).select(xldProfilelastLoginTime)
      .as(xldProfileTableAlias)
    val userProfileBuilder =  new SelectBuilder(tableName).as(userProfileTableAlias).where(SqlCondition.equalsNull(USERNAME.tableAlias(userProfileTableAlias)))
    val usersBuilder = new SelectBuilder(xlUserstableName).as(xlUsersTableAlias).where(SqlCondition.equalsNull(xlUsersUSERNAME.tableAlias(xlUsersTableAlias)))

    val joinBuilder = new JoinBuilder(selectBuilder)
      .join(
        userProfileBuilder,
        SqlCondition.equals(
          USERNAME.tableAlias(userProfileTableAlias),
          collationAppliedLowerUsername
        ),
        JoinType.Left
      )
      .join(
        usersBuilder,
        SqlCondition.equals(
          xlUsersUSERNAME.tableAlias(xlUsersTableAlias),
          collationAppliedUsername
        ),
        JoinType.Left
      )

    logger.debug("Fetching external users for insert -----> " + joinBuilder.query)
      jdbcTemplate.query(joinBuilder.query, Setter(joinBuilder.parameters), mapRowToUserProfileInsert).asScala.toList
  }

  private def mapRowToUserProfileInsert: RowMapper[UserProfile] = (rs: ResultSet, _) =>
    UserProfile(
      username = rs.getString(xldProfileUSERNAME.name),
      rs.getTimestamp(xldProfilelastLoginTime.name),
      isInternal = false
    )

  /**
   * This method fetches a list of external user profiles that need to be updated in the `XLD_USER_PROFILES` table.
   *
   * The method constructs and executes a SQL query to fetch `username` and `last active` entries from the `XLD_PROFILES` table.
   * These entries are those that have corresponding entries in `XLD_USER_PROFILES` and do not have corresponding entries in the `XL_USERS` table.
   *
   * The result is a list of `UserProfile` objects representing external users that need to be updated in the `XLD_USER_PROFILES` table.
   *
   * @return A list of `UserProfile` objects representing external users that need to be updated.
   */
  override def fetchExternalUserForUpdate(): List[UserProfile] = {
    import com.xebialabs.deployit.security.sql.XldUserProfileSchema._
    import ProfileSchema._

    val collationAppliedUsername = schemaInfo.sqlDialect.selectWithCollation(xldProfileUSERNAME.build(Some(xldProfileTableAlias)))
    val collationAppliedLowerUsername = schemaInfo.sqlDialect.selectWithCollation(SqlFunction.lower(xldProfileUSERNAME).build(Some(xldProfileTableAlias)))


    val xldProfileBuilder =  new SelectBuilder(xldProfileTableName).as(xldProfileTableAlias).select(xldProfileUSERNAME).select(xldProfilelastLoginTime)
    val usersBuilder = new SelectBuilder(xlUserstableName).as(xlUsersTableAlias).where(SqlCondition.equalsNull(xlUsersUSERNAME.tableAlias(xlUsersTableAlias)))

    val userProfileBuilder = new SelectBuilder(tableName).as(userProfileTableAlias)
    val joinBuilder = new JoinBuilder(userProfileBuilder)
      .join(
        xldProfileBuilder,
        SqlCondition.equals(
          USERNAME.tableAlias(userProfileTableAlias),
          collationAppliedLowerUsername
        ),
        JoinType.Inner
      )
      .join(
        usersBuilder,
        SqlCondition.equals(
          SqlFunction.lower(xlUsersUSERNAME.tableAlias(xlUsersTableAlias)), // We should not fetch the internal user. [xld_profiles has the two different entry for the same internal users, as internalUsers login is case insensitive: internal10_user, Internal10_USER ]
          collationAppliedLowerUsername
        ),
        JoinType.Left
      )
    logger.debug("Fetching external users for update -----> " + joinBuilder.query)
    jdbcTemplate.query(joinBuilder.query, Setter(joinBuilder.parameters), mapRowToUserProfileUpdate).asScala.toList
  }

  private def mapRowToUserProfileUpdate: RowMapper[UserProfile] = (rs: ResultSet, _) =>
    UserProfile(
      username = rs.getString(USERNAME.name),
      rs.getTimestamp(xldProfilelastLoginTime.name),
      isInternal = false
    )

  /**
   * This method fetches a list of internal user profiles that need to be updated in the `XLD_USER_PROFILES` table.
   *
   * The method constructs and executes a SQL query to fetch `username` and `last active` entries from the `XLD_PROFILES` table.
   * These entries are those that have corresponding entries in both `XLD_USER_PROFILES` and `XL_USERS` table.
   *
   * The result is a list of `UserProfile` objects representing internal users that need to be updated in the `XLD_USER_PROFILES` table.
   *
   * @return A list of `UserProfile` objects representing internal users that need to be updated.
   */
  override def fetchInternalUserForUpdate(): List[UserProfile] = {
    import com.xebialabs.deployit.security.sql.XldUserProfileSchema._
    import ProfileSchema._

    val collationAppliedUsername = schemaInfo.sqlDialect.selectWithCollation(xldProfileUSERNAME.build(Some(xldProfileTableAlias)))
    val collationAppliedLowerUsername = schemaInfo.sqlDialect.selectWithCollation(SqlFunction.lower(xldProfileUSERNAME).build(Some(xldProfileTableAlias)))

    val xldProfileBuilder =  new SelectBuilder(xldProfileTableName).as(xldProfileTableAlias).select(xldProfileUSERNAME).select(xldProfilelastLoginTime)
    val usersBuilder = new SelectBuilder(xlUserstableName).as(xlUsersTableAlias).where(SqlCondition.equalsNotNull(xlUsersUSERNAME.tableAlias(xlUsersTableAlias)))

    val userProfileBuilder = new SelectBuilder(tableName).as(userProfileTableAlias)
    val joinBuilder = new JoinBuilder(userProfileBuilder)
      .join(
        xldProfileBuilder,
        SqlCondition.equals(
          USERNAME.tableAlias(userProfileTableAlias),
          collationAppliedLowerUsername
        ),
        JoinType.Inner
      )
      .join(
        usersBuilder,
        SqlCondition.equals(
          xlUsersUSERNAME.tableAlias(xlUsersTableAlias),
          collationAppliedUsername
        ),
        JoinType.Left
      )
    logger.debug("Fetching internal users for update -----> " + joinBuilder.query)
    jdbcTemplate.query(joinBuilder.query, Setter(joinBuilder.parameters), mapRowToInternalUserProfileUpdate).asScala.toList
  }

  private def mapRowToInternalUserProfileUpdate: RowMapper[UserProfile] = (rs: ResultSet, _) =>
    UserProfile(
      username = rs.getString(USERNAME.name),
      rs.getTimestamp(xldProfilelastLoginTime.name),
    )

  /**
   * This method inserts a list of user profiles into the `XLD_USER_PROFILES` table.
   *
   * The method takes a list of `UserProfile` objects as input. Each `UserProfile` object represents a user profile that needs to be inserted into the `XLD_USER_PROFILES` table.
   * The `UserProfile` object contains the `username`, `last active` timestamp, and a flag `isInternal` indicating whether the user is an internal user.
   *
   * The method constructs an SQL insert query and executes it using `jdbcTemplate.batchUpdate`. The `UserProfile` objects are inserted into the `XLD_USER_PROFILES` table in a batch.
   *
   * @param userProfileList A list of `UserProfile` objects representing user profiles that need to be inserted into the `XLD_USER_PROFILES` table.
   */

  override def insertXldUserProfile(userProfileList: List[UserProfile]): Unit = {
    val INSERT_USER_PROFILE = sqlb"insert into $tableName ($USERNAME,$LAST_ACTIVE, $IS_INTERNAL) values (?, ?, ?)"
    val batchSize = 1000
    userProfileList.grouped(batchSize).foreach { batch =>
      if(batch.nonEmpty)
      batchInsert(INSERT_USER_PROFILE, batch)
    }
  }

  private def batchInsert(query: String, profiles: List[UserProfile]): Unit = {
    if(profiles.nonEmpty){
      jdbcTemplate.batchUpdate(query, new BatchPreparedStatementSetter {
        override def getBatchSize: Int = profiles.length

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, profiles(i).username.toLowerCase())
          ps.setTimestamp(2, profiles(i).lastActive)
          ps.setBoolean(3,profiles(i).isInternal)
        }
      })
    }
  }

  /**
   * This method updates a list of user profiles in the `XLD_USER_PROFILES` table.
   *
   * The method takes a list of `UserProfile` objects as input. Each `UserProfile` object represents a user profile that needs to be updated in the `XLD_USER_PROFILES` table.
   * The `UserProfile` object contains the `username`, `last active` timestamp, and a flag `isInternal` indicating whether the user is an internal user.
   *
   * The method constructs an SQL update query and executes it using `jdbcTemplate.batchUpdate`. The `UserProfile` objects are updated in the `XLD_USER_PROFILES` table in a batch.
   *
   * @param userProfileList A list of `UserProfile` objects representing user profiles that need to be updated in the `XLD_USER_PROFILES` table.
   */

  override def updateXldUserProfile(userProfileList: List[UserProfile]): Unit = {
    val UPDATE_USER_PROFILE = sqlb"update $tableName set  $LAST_ACTIVE = ?, $IS_INTERNAL = ? where $USERNAME = ?"
    val batchSize = 1000
    userProfileList.grouped(batchSize).foreach { batch =>
      if(batch.nonEmpty)
        batchUpdate(UPDATE_USER_PROFILE, batch)
    }
  }

  private def batchUpdate(query: String, profiles: List[UserProfile]): Unit = {
    if(profiles.nonEmpty){
      jdbcTemplate.batchUpdate(query, new BatchPreparedStatementSetter {
        override def getBatchSize: Int = profiles.length

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setTimestamp(1, profiles(i).lastActive)
          ps.setBoolean(2,profiles(i).isInternal)
          ps.setString(3, profiles(i).username.toLowerCase)
        }
      })
    }
  }

  /**
   * This method fetches all records from the `XLD_USER_PROFILES` table.
   *
   * The method constructs and executes a SQL query to fetch all entries from the `XLD_USER_PROFILES` table.
   * Each entry includes `username`, `last active` timestamp, a flag `isInternal` indicating whether the user is an internal user, and a flag `analyticsEnabled` indicating whether analytics is enabled for the user.
   *
   * The result is a list of `UserProfile` objects representing all user profiles in the `XLD_USER_PROFILES` table.
   *
   * @return A list of `UserProfile` objects representing all user profiles.
   */

  override def fetchRecords(): List[UserProfile] = {
    import com.xebialabs.deployit.security.sql.XldUserProfileSchema._
    val userProfileBuilder = new SelectBuilder(tableName).as(userProfileTableAlias).select(USERNAME).select(LAST_ACTIVE).select(IS_INTERNAL).select(ANALYTICS_ENABLED)
    jdbcTemplate.query(userProfileBuilder.query, Setter(userProfileBuilder.parameters), mapRowToUserProfileRec).asScala.toList
  }

  private def mapRowToUserProfileRec: RowMapper[UserProfile] = (rs: ResultSet, _) =>
    UserProfile(
      username = rs.getString(USERNAME.name),
      rs.getTimestamp(LAST_ACTIVE.name),
      isInternal = rs.getBoolean(IS_INTERNAL.name),
      analyticsEnabled = rs.getBoolean(ANALYTICS_ENABLED.name),
    )

}

case class UserProfile (username: String, lastActive: Timestamp = null, isInternal: Boolean = true, analyticsEnabled: Boolean = true)