package com.xebialabs.deployit.repository.sql.commands

import com.xebialabs.deployit.artifact.resolution.InternalArtifactResolver
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util._
import com.xebialabs.deployit.io.{Exploder, Imploder}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.artifact.{FolderArtifact, SourceArtifact}
import com.xebialabs.deployit.repository.ProgressLogger
import com.xebialabs.deployit.repository.sql.CiPkAndType
import com.xebialabs.deployit.repository.sql.artifacts.{ArtifactDataRepository, ArtifactRepository}
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.security.Permissions
import com.xebialabs.deployit.util.JavaCryptoUtils
import com.xebialabs.license.service.LicenseTransaction
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import org.joda.time.{DateTime, DateTimeZone}
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.net.URLEncoder
import java.security.DigestInputStream
import java.sql.ResultSet
import java.util.UUID
import scala.collection.mutable
import scala.jdk.CollectionConverters._

trait ChangeSetCommand extends CiQueries {

  def preloadCache(context: ChangeSetContext): Unit = {}

  def execute(context: ChangeSetContext): Unit

  def validate(context: ChangeSetContext): Unit

}

abstract class AbstractChangeSetCommand(jdbcTemplate: JdbcTemplate)
  extends ChangeSetCommand {

  protected def findPkfor(item: ConfigurationItem, context: ChangeSetContext): CiPKType = {
    item.get$internalId() match {
      case internalId: Integer =>
        context.pkCache.put(item, internalId)
        internalId
      case _ =>
        try {
          context.pkCache.getOrElseUpdate(item, jdbcTemplate.queryForObject(SELECT_ID_BY_PATH, classOf[CiPKType], idToPath(item.getId)))
        } catch {
          case _: EmptyResultDataAccessException =>
            throw new Checks.IncorrectArgumentException("The id [%s] does not exist.", item.getId)
        }
    }
  }

  protected def findCiForId(id: String, context: ChangeSetContext): CiPkAndType = findCiForPath(idToPath(id), context)

  protected def findCiForPath(path: String, context: ChangeSetContext): CiPkAndType = {
    context.pathCache.getOrElseUpdate(path, jdbcTemplate.query(SELECT_CI_BY_PATH, MapRowMapper, path).asScala.headOption
      .map(m => CiPkAndType(m))
      .getOrElse(throw new Checks.IncorrectArgumentException("The path [%s] does not exist.", path)))
  }
}

abstract class AbstractInsertOrUpdateCommand(jdbcTemplate: JdbcTemplate,
                                             artifactRepository: ArtifactRepository,
                                             artifactDataRepository: ArtifactDataRepository,
                                             checksumAlgorithmProvider: ChecksumAlgorithmProvider)
  extends AbstractChangeSetCommand(jdbcTemplate) {

  private def nextUUID: String = UUID.randomUUID().toString

  protected def generateToken: String = nextUUID

  protected def generateDirectoryId: String = nextUUID

  private def transformCiDataResponse[T](items: Seq[ConfigurationItem],
                                         ciData: Map[String, T],
                                         extractCiPkFromDBResponse: T => CiPKType,
                                         extractResultFromCi: (ConfigurationItem, T) => T,
                                         context: ChangeSetContext): Map[ConfigurationItem, T] = {
    items.flatMap {
      case ci if ciData.contains(ci.getId) =>
        ciData.get(ci.getId).map { data =>
          context.pkCache.put(ci, extractCiPkFromDBResponse(data))
          ci -> data
        }
      case ci if context.pkCache.contains(ci) => context.pkCache.get(ci).map(pk => ci -> extractResultFromCi(ci, pk.asInstanceOf[T]))
      case ci => throw new Checks.IncorrectArgumentException("The id [%s] does not exist.", ci.getId)
    }.toMap
  }

  private def loadPksByPaths(paths: Seq[String]): Map[String, CiPKType] =
    queryWithInClause(paths) { group =>
      jdbcTemplate.query(buildSelectIdsByPathsQuery(group), Setter(group), new RowMapper[(String, CiPKType)] {
        override def mapRow(rs: ResultSet, rowNum: Int): (String, CiPKType) =
          pathToId(rs.getString(CIS.path.name)) -> asCiPKType(rs.getInt(CIS.ID.name))
      }).asScala
    }.toMap

  protected def findPksfor(items: Seq[ConfigurationItem], context: ChangeSetContext): Map[ConfigurationItem, CiPKType] = {
    val pathToLookup = items.to(LazyList).filterNot(context.pkCache.contains).map(_.getId).map(idToPath)
    val loadedPks = loadPksByPaths(pathToLookup)
    transformCiDataResponse[CiPKType](items, loadedPks, identity, (_, pk) => pk, context)
  }

  private def loadPksAndTokensByPaths(paths: Seq[String]): Map[String, (CiPKType, String)] =
    queryWithInClause(paths) { group =>
      jdbcTemplate.query(buildSelectIdsAndTokensByPathsQuery(group), Setter(group), new RowMapper[(String, (CiPKType, String))] {
        override def mapRow(rs: ResultSet, rowNum: Int): (String, (CiPKType, String)) =
          pathToId(rs.getString(CIS.path.name)) -> (asCiPKType(rs.getInt(CIS.ID.name)), rs.getString(CIS.token.name))
      }).asScala
    }.toMap

  protected def findCiPksAndTokensFor(items: Seq[ConfigurationItem], context: ChangeSetContext): Map[ConfigurationItem, (CiPKType, String)] = {
    val pathsAndTokens = loadPksAndTokensByPaths(items.to(LazyList).map(_.getId).map(idToPath))
    transformCiDataResponse[(CiPKType, String)](items, pathsAndTokens, { case (pk, _) => pk }, (_, data) => data, context)
  }

  protected def handleSourceArtifactFilename(pk: CiPKType, sourceArtifact: SourceArtifact): Unit = {
    artifactRepository.updateFilename(pk, sourceArtifact.getFile.getName)
  }

  protected def handleSourceArtifactFileData(pk: CiPKType, sourceArtifact: SourceArtifact): Unit = {
    val file = sourceArtifact.getFile
    val digest = checksumAlgorithmProvider.getMessageDigest
    using(sourceArtifact match {
      case _: FolderArtifact if file.isDirectory =>
        Imploder.implode(file, digest).get()
      case _: FolderArtifact =>
        Exploder.calculateCheckSum(() => file.getInputStream, file.getName, digest)
        file.getInputStream
      case _ =>
        new DigestInputStream(file.getInputStream, digest)
    }) {
      artifactDataRepository.store(pk, _)
    }
    val checksum = JavaCryptoUtils.digest(digest)
    if (sourceArtifact.getChecksum == null)
      sourceArtifact.setProperty(SourceArtifact.CHECKSUM_PROPERTY_NAME, checksum)
    if (sourceArtifact.getFileUri == null)
      sourceArtifact.setProperty(SourceArtifact.FILE_URI_PROPERTY_NAME, s"${InternalArtifactResolver.Protocol}:${URLEncoder.encode(file.getName, "UTF-8")}")
  }
}

class ChangeSetContext(val licenseTransaction: LicenseTransaction,
                       progressLogger: ProgressLogger,
                       val scmTraceabilityData: Option[SCMTraceabilityData] = None) {
  val now: DateTime = DateTime.now(DateTimeZone.UTC)
  val userName: String = Permissions.getAuthenticatedUserName
  val pkCache: mutable.Map[ConfigurationItem, CiPKType] = mutable.Map()
  val pathCache: mutable.Map[String, CiPkAndType] = mutable.Map()
  val ciCache: mutable.Map[String, ConfigurationItem] = mutable.Map()

  def log(output: String, params: AnyRef*): Unit = progressLogger.log(output, params: _*)
}
