package com.xebialabs.deployit.repository.sql.specific.configurable

import com.xebialabs.deployit.core.sql.batch.BatchCommandWithSetter
import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.repository.sql.reader.properties.CiDataProvider
import com.xebialabs.deployit.repository.sql.specific._
import com.xebialabs.deployit.repository.sql.specific.columns._
import com.xebialabs.deployit.repository.sql.specific.configurable.ConfigurableTypeSpecificPersisterFactory.CiFromMapReader
import org.springframework.jdbc.core.JdbcTemplate

import java.util.{Map => JMap, List => JList}
import scala.jdk.CollectionConverters._
import com.xebialabs.deployit.core.util.CiSugar._

object ConfigurableTypeSpecificPersisterFactory {
  type CiFromMapReader = (JMap[String, Object], CiDataProvider) => ConfigurationItem
}

abstract class ConfigurableTypeSpecificPersisterFactory(typeClass: Class[_]) extends AbstractTypeSpecificPersisterFactory(typeClass) {
  val config: TypeConfiguration

  override def createReader(pk: CiPKType): TypeSpecificReader = new ConfigurableReader(pk,
    config.mainTable.map(_.reader),
    config.propertyTables
  )

  override def createInserter(pk: CiPKType): TypeSpecificInserter = new ConfigurableInserter(pk,
    config.mainTable.map(_.inserter),
    config.propertyTables.view.mapValues(_.inserter).toMap
  )

  override def createUpdater(pk: CiPKType): TypeSpecificUpdater = new ConfigurableUpdater(pk,
    config.mainTable.map(_.updater),
    config.propertyTables.view.mapValues(_.updater).toMap
  )

  override def createDeleter(pk: CiPKType): TypeSpecificDeleter = new ConfigurableDeleter(pk,
    config.mainTable.map(_.deleter),
    config.propertyTables.values.map(_.deleter).toList
  )

  override def createReferenceFinder(pks: Seq[CiPKType]): TypeSpecificReferenceFinder = new ConfigurableReferenceFinder(pks,
    config.mainTable.map(_.referenceFinder(Type.valueOf(typeClass))),
    config.propertyTables.values.map(_.referenceFinder).toList
  )
}

class TypeConfiguration(val mainTable: Option[PropertyColumnsTable], val propertyTables: Map[String, PropertyTable[_, _]] = Map())

class PropertyColumnsTable(val table: TableName, val idColumn: ColumnName, val columns: Map[String, ColumnName] = Map())
                          (implicit val jdbcTemplate: JdbcTemplate, implicit val schemaInfo: SchemaInfo) {
  val reader = new PropertyColumnsReader(table, idColumn, columns)
  val inserter = new PropertyColumnsInserter(table, idColumn, columns)
  val updater = new PropertyColumnsUpdater(table, idColumn, columns)
  val deleter = new PropertyColumnsDeleter(table, idColumn)

  def referenceFinder(ciType: Type): PropertyColumnsReferenceFinder = {
    val types = DescriptorRegistry.getSubtypes(ciType).asScala.toList :+ ciType
    new PropertyColumnsReferenceFinder(
      table,
      idColumn,
      columns.filter(column =>
        types.exists(t => t.getDescriptor.getPropertyDescriptor(column._1) != null
          && t.getDescriptor.getPropertyDescriptor(column._1).getKind == PropertyKind.CI)
      ).values.toList
    )
  }
}

case class PropertyTableSelectDescriptor(tableName: TableName,
                                         selectColumns: List[ColumnName],
                                         ciIdColumn: ColumnName,
                                         orderColumns: List[OrderBy] = List(),
                                         ciRefColumn: Option[ColumnName] = None)

trait PropertyTable[T, V] {
  implicit val selectDescriptor: PropertyTableSelectDescriptor
  val reader: PropertyTableReader[V]
  val inserter: PropertyTableInserter[T]
  val updater: PropertyTableUpdater[T]
  val deleter: PropertyTableDeleter
  val referenceFinder: PropertyTableReferenceFinder
}

trait PropertyTableReader[V] {
  def readProperty(pk: CiPKType)(implicit ciDataProvider: CiDataProvider,
                                 readFromMap: CiFromMapReader,
                                 typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory]): V

  protected def retrieveMaps(pk: CiPKType)(implicit ciDataProvider: CiDataProvider,
                                           selectDescriptor: PropertyTableSelectDescriptor): (JList[JMap[String, AnyRef]], Seq[(CiPKType, Type)]) = {
    val values = ciDataProvider.getPropertyTableValues(pk, selectDescriptor)
    val ciPkAndTypes = selectDescriptor
      .ciRefColumn
      .map(_ => values.asScala.map(map => asCiPKType(map.get(CIS.ID.name)) -> map.get(CIS.ci_type.name).toString.ciType))
      .getOrElse(Nil)
    (values, ciPkAndTypes.toSeq)
  }
}

trait PropertyTableInserter[T] {
  def insertProperty(pk: CiPKType, value: Any, handler: ErrorHandler): Unit = insertTypedProperty(pk, value.asInstanceOf[T], handler)

  def insertTypedProperty(pk: CiPKType, value: T, handler: ErrorHandler): Unit

  def batchInsertProperty(pk: CiPKType, value: Any): List[BatchCommandWithSetter] = batchInsertTypedProperty(pk, value.asInstanceOf[T])

  def batchInsertTypedProperty(pk: CiPKType, value: T): List[BatchCommandWithSetter]
}

trait PropertyTableUpdater[T] {
  def updateProperty(pk: CiPKType, value: Any)(implicit ciDataProvider: CiDataProvider): Unit =
    updateTypedProperty(pk, value.asInstanceOf[T])

  def updateTypedProperty(pk: CiPKType, t: T)(implicit ciDataProvider: CiDataProvider): Unit

  def batchUpdateProperty(pk: CiPKType, value: Any)(implicit ciDataProvider: CiDataProvider): List[BatchCommandWithSetter] =
    batchUpdateTypedProperty(pk, value.asInstanceOf[T])

  def batchUpdateTypedProperty(pk: CiPKType, t: T)(implicit ciDataProvider: CiDataProvider): List[BatchCommandWithSetter]
}

trait PropertyTableDeleter {
  def deleteProperties(pk: CiPKType): Unit
}

trait PropertyTableReferenceFinder {
  def findReferences(pks: Seq[CiPKType]): Seq[(CiPKType, CiPKType)]

  def findReferencesBuilder(pks: Seq[CiPKType]): Option[SelectBuilder]
}
