package com.xebialabs.deployit.repository.sql.commands

import com.xebialabs.deployit.artifact.resolution.InternalArtifactResolver
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs, BatchCommandWithSetter, BatchExecutorRepository}
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.engine.spi.event.CisUpdatedEvent
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.io.ArtifactFileUtils.hasRealOrResolvedFile
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, ExternalProperty}
import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.deployit.repository.sql.artifacts.{ArtifactDataRepository, ArtifactRepository}
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.properties.{DeleteRepositoryCiProperties, PropertyManipulator, UpdateCiPropertiesRepository}
import com.xebialabs.deployit.repository.sql.reader.properties.{CiDataProvider, CiGroupCiDataProvider, OnDemandCiDataProvider}
import com.xebialabs.deployit.repository.sql.specific.{TypeSpecificPersisterFactory, TypeSpecificUpdater}
import com.xebialabs.deployit.repository.sql.{CiHistoryRepository, CiRepository}
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry
import com.xebialabs.xlplatform.coc.service.{PersistenceParams, SCMTraceabilityService}
import org.springframework.jdbc.core.JdbcTemplate

import java.lang.Long
import java.sql.Statement
import java.util.{Collections, ArrayList => JavaArrayList, Collection => JavaCollection, HashMap => JavaHashMap, List => JList, Map => JavaMap}
import scala.jdk.CollectionConverters._

class UpdateCommand(override val jdbcTemplate: JdbcTemplate,
                    artifactRepository: ArtifactRepository,
                    artifactDataRepository: ArtifactDataRepository,
                    override val ciRepository: CiRepository,
                    ciHistoryRepository: CiHistoryRepository,
                    batchExecutorRepository: BatchExecutorRepository,
                    passwordEncrypter: PasswordEncrypter,
                    createTypeSpecificUpdaters: (Type, CiPKType) => List[TypeSpecificUpdater],
                    scmTraceabilityService: SCMTraceabilityService,
                    val cis: Seq[ConfigurationItem],
                    checksumAlgorithmProvider: ChecksumAlgorithmProvider)
                   (implicit val schemaInfo: SchemaInfo, typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory])
  extends AbstractInsertOrUpdateCommand(jdbcTemplate, artifactRepository, artifactDataRepository, checksumAlgorithmProvider)
    with UpdateCiPropertiesRepository
    with DeleteRepositoryCiProperties
    with UpdateCommandQueries
    with LookupValuesQueries
    with CiPreloadCache {

  private implicit val template: JdbcTemplate = jdbcTemplate

  override def preloadCache(context: ChangeSetContext): Unit =
    if (cis.nonEmpty) {
      preloadCacheForCis(cis, context)
    }

  override def execute(context: ChangeSetContext): Unit =
    if (cis.nonEmpty) {
      update(context)
      EventBusHolder.publish(new CisUpdatedEvent(context.scmTraceabilityData.orNull, cis.toList.asJava))
    }

  override def validate(context: ChangeSetContext): Unit = validateUpdate(cis, context)

  private def saveTraceabilityData(ci: ConfigurationItem, context: ChangeSetContext): Any = {
    ci match {
      case baseCi: BaseConfigurationItem =>
        val id = scmTraceabilityService.persistOrDelete(PersistenceParams(
          Option(baseCi.get$ciAttributes().getScmTraceabilityDataId),
          context.scmTraceabilityData
        ))
        baseCi.get$ciAttributes().setScmTraceabilityDataId(id.orNull)
        id.getOrElse(Setter.NULL_INTEGER)
      case _ => Setter.NULL_INTEGER
    }
  }

  private def update(context: ChangeSetContext): Unit = {

    val (batchUpdateTokens, batchUpdateCis) = updateCis(context)
    val updateTokensResult = batchExecutorRepository.executeWithSetter(batchUpdateTokens)
    validateUpdateTokensResult(updateTokensResult, context)
    batchExecutorRepository.executeWithSetter(batchUpdateCis)

    val (batchExternalPropertyChanges, batchExternalPropertyInserts) = updateExternalProperties(context)
    batchExecutorRepository.execute(batchExternalPropertyChanges)
    batchExecutorRepository.execute(batchExternalPropertyInserts)

    val batchCiHistory = updateCiHistory(context)
    batchExecutorRepository.execute(batchCiHistory)

    val (batchGenericProps, batchTypeSpecificProps) = updateCiProperties(context)
    batchExecutorRepository.execute(batchGenericProps)
    batchExecutorRepository.executeWithSetter(batchTypeSpecificProps)
  }

  private[commands] def updateCis(context: ChangeSetContext): (Iterable[BatchCommandWithSetter], Iterable[BatchCommandWithSetter]) = {
    val init = (Vector[BatchCommandWithSetter](), Vector[BatchCommandWithSetter]())
	var previousUpdate:(Integer, String) = (-1, null)
	
    cis.foldLeft(init) { case ((tokenUpdates, ciUpdates), ci) =>
      context.log("Updating %s of type %s", ci.getId, ci.getType)

      val ciPk = findCiForId(ci.getId, context)
      val token = generateToken

      val traceabilityId = saveTraceabilityData(ci, context)

      val (updateCi, updateToken) = ci match {
        case bci: BaseConfigurationItem if bci.get$token() != null =>
          val(previousKey, previousToken) = previousUpdate
          val whereClauseToken = if(!previousKey.equals(-1) && previousKey.equals(ciPk.pk) && previousToken != null){
            previousToken
          } else {
            bci.get$token()
          }
          (None, Some(BatchCommand(UPDATE_TOKEN, Setter(Array[Any](token, context.now, context.userName, traceabilityId, ciPk.pk, whereClauseToken)))))
        case _ =>
          (Some(BatchCommand(UPDATE, Setter(Array[Any](token, context.now, context.userName, traceabilityId, ciPk.pk)))), None)
      }

      asBaseConfigurationItem(ci) { bci =>
        bci.set$token(token)
        bci.set$internalId(ciPk.pk)
      }

	  previousUpdate = (ciPk.pk, token)
      (tokenUpdates ++ updateToken, ciUpdates ++ updateCi)
    }
  }

  private[commands] def validateUpdateTokensResult(results: Iterable[Array[Int]], context: ChangeSetContext): Unit = {
    val allResults = results.flatten
    val cannotCount = allResults.count(count => {
      if (count >= 0 && count != 1)
        throw new ItemConflictException("Repository entity has been updated since you read it. Please reload the CI from the repository again.")
      count == Statement.SUCCESS_NO_INFO
    })
    if (cannotCount > 0) {
      val sumCounts = cis
        .flatMap {
          case bci: BaseConfigurationItem => Option(bci.get$token()).map((findCiForId(bci.getId, context).pk, _))
          case _ => None
        }
        .grouped(schemaInfo.sqlDialect.inClauseLimit)
        .foldLeft(0) { (sum, pairs) =>
          val pks = pairs.map(_._1)
          val tokens = pairs.map(_._2)
          val selectBuilder: SelectBuilder = new SelectBuilder(CIS.tableName).select(new SqlLiteral("*")).count()
            .where(cond.and(Seq(cond.in(CIS.ID, pks), cond.in(CIS.token, tokens))))
          sum + jdbcTemplate.queryForObject(selectBuilder.query, selectBuilder.parameters.map(_.asInstanceOf[AnyRef]).toArray, classOf[Number])
            .intValue
        }
      if (sumCounts != allResults.size)
        throw new ItemConflictException("Repository entity has been updated since you read it. Please reload the CI from the repository again.")
    }
  }

  private[commands] def updateCiHistory(context: ChangeSetContext): Iterable[BatchCommandWithArgs] =
    cis.flatMap { ci =>
      val ciPk = findCiForId(ci.getId, context)
      val ciHistoryBatch = ciHistoryRepository.batchInsert(ciPk.pk, ci, context.now, context.userName)
      updateSourceArtifact(ciPk.pk, ci)
      ciHistoryBatch
    }

  private def updateCiProperties(context: ChangeSetContext): (Vector[BatchCommandWithArgs], Vector[BatchCommandWithSetter]) = {
    val ciPksAndTypes = cis.map(ci => findCiForId(ci.getId, context).pk -> ci.getType)
    implicit val dataProvider: CiGroupCiDataProvider = new CiGroupCiDataProvider(ciPksAndTypes)
    cis.foldLeft((Vector.empty[BatchCommandWithArgs], Vector.empty[BatchCommandWithSetter])) { case ((withArgs, withSetter), ci) =>
      val pk = findCiForId(ci.getId, context).pk
      val (allGeneric, allTypeSpecific) = new DefaultPropertyUpdater(ci, pk, dataProvider, passwordEncrypter)
        .updateProperties(context)

      (withArgs ++ allGeneric, withSetter ++ allTypeSpecific)
    }
  }

  private def validateUpdate(cis: Seq[ConfigurationItem], context: ChangeSetContext): Unit = {
    findCiPksAndTokensFor(cis, context)
      .foreach { case (ci, (pk, token)) =>
        asBaseConfigurationItem(ci) { bci =>
          if (Option(bci.get$token()).exists(_ != token)) {
            throw new ItemConflictException("Repository entity [%s] has been updated since you read it. Please reload the CI from the repository again.", bci.getId)
          }
        }
        new DefaultPropertyUpdater(ci, pk, new OnDemandCiDataProvider, passwordEncrypter).validateProperties()
      }
  }

  private def updateSourceArtifact(pk: CiPKType, item: ConfigurationItem): Unit =
    item match {
      case sa: SourceArtifact if hasRealOrResolvedFile(sa) && (sa.getFileUri == null || InternalArtifactResolver.canHandle(sa.getFileUri)) =>
        // Internal file: store filename and file data.
        if (ArtifactResolverRegistry.isStoredArtifact(sa))
          artifactDataRepository.delete(pk)
        handleSourceArtifactFilename(pk, sa)
        handleSourceArtifactFileData(pk, sa)
      case sa: SourceArtifact if hasRealOrResolvedFile(sa) =>
        // External file: store filename
        handleSourceArtifactFilename(pk, sa)
        sa.setFile(null)
      case _ =>
      // do nothing
    }

  private[commands] def updateExternalProperties(context: ChangeSetContext): (Vector[BatchCommandWithArgs], Vector[BatchCommandWithArgs]) = {
    val ciPksAndTypes = cis.map(ci => findCiForId(ci.getId, context).pk -> ci.getType)
    val ciDataProvider = new CiGroupCiDataProvider(ciPksAndTypes)
    cis.foldLeft((Vector.empty[BatchCommandWithArgs], Vector.empty[BatchCommandWithArgs])) { case ((updates, inserts), ci) =>
      Option(asBaseConfigurationItem(ci) { bci =>
        val pk = findCiForId(ci.getId, context).pk
        val externalProperties = new JavaHashMap[String, ExternalProperty](bci.get$externalProperties)

        val updateLookupValues = ciDataProvider.getLookupValues(pk).asScala.flatMap { map =>
          val propertyName = map.get(LOOKUP_VALUES.name.name).asInstanceOf[String]
          Option(externalProperties.remove(propertyName)) match {
            case Some(newValue: LookupValueKey) =>
              val oldKey = map.get(LOOKUP_VALUES.key.name)
              val oldProviderId = asCiPKType(map.get(LOOKUP_VALUES.provider.name))
              val newKey = newValue.getKey
              val newProviderId = findCiForId(newValue.getProviderId, context).pk
              Option((newKey, newProviderId))
                .filter(_ != (oldKey, oldProviderId))
                .map { case (key, providerId) => BatchCommand(UPDATE_LOOKUP_VALUE, key, providerId, pk, propertyName) }
            case _ => Some(BatchCommand(DELETE_LOOKUP_VALUE, pk, propertyName))
          }
        }

        val insertLookupValues = externalProperties.asScala.map {
          case (name, lookup: LookupValueKey) =>
            BatchCommand(INSERT_LOOKUP_VALUES, pk, name, lookup.getKey, findCiForId(lookup.getProviderId, context).pk)
          case (_, e: ExternalProperty) => throw new IllegalArgumentException(s"Unknown external property $e.");
        }

        (updates ++ updateLookupValues, inserts ++ insertLookupValues)
      }).getOrElse((Vector.empty, Vector.empty))
    }
  }

  private class DefaultPropertyUpdater(val ci: ConfigurationItem,
                                       val pk: CiPKType,
                                       val ciDataProvider: CiDataProvider,
                                       override val passwordEncrypter: PasswordEncrypter)
    extends PropertyManipulator {

    def nullValueFor(kind: PropertyKind): Any = kind match {
      case PropertyKind.INTEGER | PropertyKind.CI => Setter.NULL_INTEGER
      case PropertyKind.BOOLEAN => Setter.NULL_BOOLEAN
      case PropertyKind.DATE => Setter.NULL_DATETIME
      case _ => null
    }

    def updateProperties(context: ChangeSetContext)
                        (implicit dataProvider: CiDataProvider): (Seq[BatchCommandWithArgs], Seq[BatchCommandWithSetter]) = {
      val properties = ciDataProvider.getProperties(pk)
      val map = properties.asScala.groupBy(_.asScala(CI_PROPERTIES.name.name))
      val updaters = createTypeSpecificUpdaters(ci.getType, pk)

      validateProperties()

      val allGenericCommands: Seq[BatchCommandWithArgs] = listNonTransientProperties(ci.getType).flatMap { pd =>

        val value =
          if (isExternal(pd, ci))
            None
          else getValue(ci, pd, ci => findCiForId(ci.getId, context).pk)

        whenUpdatersDoNotHandle(updaters, pd, value.getOrElse(nullValueFor(pd.getKind))) {
          (value, map.get(pd.getName)) match {
            case (None, None) => Iterable.empty // No Action
            case (None, Some(oldValue)) =>
              oldValue.map(getPropertyPK).map(batchDeleteProperty)
            case (Some(newValue), None) =>
              pd.getKind match {
                case k if k.isSimple =>
                  Iterable(batchInsertProperty(pk, pd.getName, pd.getKind, newValue))
                case PropertyKind.CI =>
                  if (!pd.isAsContainment) // ignore as-containment properties
                    Iterable(batchInsertProperty(pk, pd.getName, pd.getKind, newValue))
                  else
                    Iterable.empty
                case PropertyKind.SET_OF_CI =>
                  if (!pd.isAsContainment) // ignore as-containment properties
                    batchInsertIndexedProperty(pk, pd.getName, pd.getKind, newValue.asInstanceOf[JavaCollection[String]])
                  else
                    Iterable.empty
                case PropertyKind.LIST_OF_CI | PropertyKind.LIST_OF_STRING | PropertyKind.SET_OF_STRING =>
                  batchInsertIndexedProperty(pk, pd.getName, pd.getKind, newValue.asInstanceOf[JavaCollection[String]])
                case PropertyKind.MAP_STRING_STRING =>
                  batchInsertKeyedProperty(pk, pd.getName, pd.getKind, newValue.asInstanceOf[JavaMap[String, String]])
              }
            case (Some(newValue), Some(oldValue)) =>
              pd.getKind match {
                case k if k.isSimple =>
                  val map = oldValue.headOption.getOrElse(Collections.emptyMap())
                  if (newValue != getSimpleValue(k, map)) {
                    Iterable(batchUpdateProperty(getPropertyPK(map), k, newValue))
                  } else
                    Iterable.empty
                case PropertyKind.CI =>
                  if (!pd.isAsContainment) { // ignore as-containment properties
                    val map = oldValue.headOption.getOrElse(Collections.emptyMap())
                    if (newValue != getCiRefValue(map)) {
                      Iterable(batchUpdateProperty(getPropertyPK(map), pd.getKind, newValue))
                    } else
                      Iterable.empty
                  } else
                    Iterable.empty
                case PropertyKind.SET_OF_CI =>
                  if (!pd.isAsContainment) { // ignore as-containment properties
                    batchUpdateListProperty(pd, newValue.asInstanceOf[JavaCollection[CiPKType]], oldValue.toSeq)
                  } else
                    Iterable.empty
                case PropertyKind.LIST_OF_CI =>
                  val containmentPropertiesNotInList =
                    if (pd.getKind == PropertyKind.LIST_OF_CI && pd.isAsContainment) {
                      addAsContainmentPropertiesNotInList(pk, pd, newValue.asInstanceOf[JavaCollection[CiPKType]])
                    } else
                      Iterable.empty
                  containmentPropertiesNotInList ++ batchUpdateListProperty(pd, newValue.asInstanceOf[JavaCollection[CiPKType]], oldValue.toSeq)
                case PropertyKind.LIST_OF_STRING | PropertyKind.SET_OF_STRING =>
                  batchUpdateListProperty(pd, newValue.asInstanceOf[JavaCollection[String]], oldValue.toSeq)
                case PropertyKind.MAP_STRING_STRING =>
                  batchUpdateMapProperty(pd, newValue.asInstanceOf[JavaMap[String, String]], oldValue.toSeq)
              }
          }
        }
      }

      (allGenericCommands, updaters.flatMap(_.batchFinish()))
    }

    def validateProperties(): Unit = {
      forAllNonTransientProperties(ci.getType) { pd: PropertyDescriptor =>
        pd.getKind match {
          case PropertyKind.LIST_OF_CI if pd.isAsContainment =>
            verifyAsContainmentCiList(ci, pd.get(ci).asInstanceOf[JavaCollection[ConfigurationItem]], pd.getName)
          case _ =>
        }
      }
    }

    private def addAsContainmentPropertiesNotInList(pk: CiPKType, pd: PropertyDescriptor, currentList: JavaCollection[CiPKType]): Iterable[BatchCommandWithArgs] = {
      jdbcTemplate.query(SELECT_CI_BY_PARENT_ID, MapRowMapper, pk).asScala.flatMap { map =>
        val childType = Type.valueOf(map.get(CIS.ci_type.name).asInstanceOf[String])
        Option(asCiPKType(map.get(CIS.ID.name)))
          .filter(childType.instanceOf(pd.getReferencedType) && !currentList.contains(_))
          .map(batchInsertIndexedCiPropertyLastPosition(pk, pd.getName, _))
      }
    }

    private def verifyAsContainmentCiList(parent: ConfigurationItem, children: JavaCollection[ConfigurationItem], propertyName: String): Unit = {
      children.forEach { child =>
        if (!child.getId.startsWith(parent.getId + '/'))
          throw new Checks.IncorrectArgumentException("Cannot add [%s] to property [%s] of item [%s], as it is not contained by it.", child.getId, propertyName, parent.getId)
      }
    }

    private def batchUpdateListProperty[T <: Object](pd: PropertyDescriptor, newValues: JavaCollection[T], oldValue: Seq[JavaMap[String, AnyRef]]): Iterable[BatchCommandWithArgs] = {
      val valueCopy = new JavaArrayList(newValues)
      val newSize = valueCopy.size
      val oldSize = oldValue.size
      val updates = oldValue.flatMap { map =>
        val idx = getIndexValue(map)
        if (idx < newSize) {
          Option(valueCopy.get(idx))
            .filter(_ != getSimpleValue(pd.getKind, map))
            .map(batchUpdateProperty(getPropertyPK(map), pd.getKind, _))
        } else {
          Some(batchDeleteProperty(getPropertyPK(map)))
        }
      }
      if (newSize > oldSize) {
        updates ++ batchInsertIndexedProperty(pk, pd.getName, pd.getKind, valueCopy.subList(oldSize, newSize), oldSize)
      } else
        updates
    }

    private def batchUpdateMapProperty(pd: PropertyDescriptor, newValues: JavaMap[String, String], oldValue: Seq[JavaMap[String, AnyRef]]):
    Iterable[BatchCommandWithArgs] = {
      val valueCopy = new JavaHashMap[String, String](newValues)
      val updates = oldValue.flatMap { map =>
        Option(valueCopy.remove(map.get(CI_PROPERTIES.key.name))) match {
          case None => Some(batchDeleteProperty(getPropertyPK(map)))
          case Some(v) if v != getStringValue(map) => Some(batchUpdateProperty(getPropertyPK(map), pd.getKind, v))
          case _ => None
        }
      }
      updates ++ batchInsertKeyedProperty(pk, pd.getName, pd.getKind, valueCopy)
    }

    private def getPropertyPK(map: JavaMap[String, AnyRef]): Long =
      asLong(map.get(CI_PROPERTIES.ID.name))

    private def whenUpdatersDoNotHandle(updaters: List[TypeSpecificUpdater], pd: PropertyDescriptor, value: Any)
                                       (block: => Iterable[BatchCommandWithArgs]): Iterable[BatchCommandWithArgs] = {
      if (updaters.isEmpty || updaters.forall(updater => !updater.updateProperty(pd.getName, value)))
        block
      else
        Iterable.empty
    }
  }

}

trait UpdateCommandQueries extends Queries {
  lazy val UPDATE: String = {
    import CIS._
    sqlb"update $tableName set $token = ?, $modified_at = ?, $modified_by = ?, $scm_traceability_data_id = ? where $ID = ?"
  }

  lazy val UPDATE_TOKEN: String = {
    import CIS._
    sqlb"update $tableName set $token = ?, $modified_at = ?, $modified_by = ?, $scm_traceability_data_id = ? where $ID = ? and $token = ?"
  }

  lazy val SELECT_ID_BY_ID_AND_TOKEN: String = {
    import CIS._
    sqlb"select $ID from $tableName where $ID = ? and $token = ?"
  }
}
