package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.sql.asInteger
import com.xebialabs.deployit.core.util.CollectionUtil
import com.xebialabs.deployit.core.{ListOfStringView, MapStringStringView, SetOfStringView, StringValue}
import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, PropertyDescriptor, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.repository.StringValueConverter._
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.deployit.sql.base.schema.CIS

import java.lang.String.format
import java.util
import scala.jdk.CollectionConverters._

package object base {
  type CiPKType = Integer

  val asCiPKType: Any => CiPKType = asInteger

  def idToPath(id: String): String = {
    val path = if (id.charAt(0) != '/') s"/$id" else id
    if (path.takeRight(1) == "/") path.dropRight(1) else path
  }

  def parentPath(id: String): Option[String] = if (id.indexOf('/') == -1) None else
    Some(idToPath(id)).map(path => path.substring(0, path.lastIndexOf('/')))

  def pathToId(path: String): String = path.substring(1)

  def asBaseConfigurationItem[T](item: ConfigurationItem)(block: BaseConfigurationItem => T): T =
    item match {
      case bi: BaseConfigurationItem => block(bi)
      case _ => null.asInstanceOf[T]
    }

  def readBaseCiFromMap[T <: ConfigurationItem](ciMap: util.Map[String, AnyRef]): ConfigurationItem = {
    val ciId = pathToId(ciMap.get(CIS.path.name).asInstanceOf[String])
    val ciType = Type.valueOf(ciMap.get(CIS.ci_type.name).asInstanceOf[String])
    if (!ciType.exists) {
      throw new TypeNotFoundException(format("Unknown type [%s] while reading node [%s]", ciType, ciId))
    }
    val descriptor = DescriptorRegistry.getDescriptor(ciType)
    val ci: ConfigurationItem = descriptor.newInstance(ciId)
    asBaseConfigurationItem(ci) { baseCi =>
      val pk = asCiPKType(ciMap.get(CIS.ID.name))
      baseCi.set$internalId(pk)
    }
    ci
  }

  def getBaseCi[T <: ConfigurationItem](ciPath: String, ciType: Type, ciPkId: Int): ConfigurationItem = {
    val ciId = pathToId(ciPath)
    if (!ciType.exists) {
      throw new TypeNotFoundException(format("Unknown type [%s] while reading node [%s]", ciType, ciId))
    }
    val descriptor = DescriptorRegistry.getDescriptor(ciType)
    val ci: ConfigurationItem = descriptor.newInstance(ciId)
    asBaseConfigurationItem(ci) { baseCi =>
      val pk = asCiPKType(ciPkId)
      baseCi.set$internalId(pk)
    }
    ci
  }

  def isTransientOrExternal(pd: PropertyDescriptor, ci: ConfigurationItem): Boolean =
    pd.isTransient || isExternal(pd, ci)

  def isExternal(pd: PropertyDescriptor, ci: ConfigurationItem): Boolean =
    asBaseConfigurationItem(ci)(_.get$externalProperties().containsKey(pd.getName))

  def forAllNonTransientProperties(t: Type, filter: PropertyDescriptor => Boolean = !_.isTransient)(block: PropertyDescriptor => Unit): Unit = {
    import scala.compat.java8.FunctionConverters._
    t.getDescriptor.getPropertyDescriptors.stream.filter(filter.asJava).forEach(block(_))
  }

  def listNonTransientProperties(t: Type, filter: PropertyDescriptor => Boolean = !_.isTransient): List[PropertyDescriptor] =
    t.getDescriptor.getPropertyDescriptors.asScala.toList.filter(filter)

  private[sql] def convertList(value: util.List[String], encrypt: Boolean, passwordEncrypter: PasswordEncrypter) = {
    val result = new ListOfStringView(CollectionUtil.apply[String, StringValue](value, stringToValue(passwordEncrypter)))
    if (encrypt)
      result.encrypt
    else
      result
  }

  private[sql] def convertSet(value: util.Set[String], encrypt: Boolean, passwordEncrypter: PasswordEncrypter) = {
    val result = new SetOfStringView(CollectionUtil.apply[String, StringValue](value, stringToValue(passwordEncrypter)))
    if (encrypt)
      result.encrypt
    else
      result
  }

  private[sql] def convertMap(value: util.Map[String, String], encrypt: Boolean, passwordEncrypter: PasswordEncrypter) = {
    val result = new MapStringStringView(CollectionUtil.apply[String, StringValue](value, stringToValue(passwordEncrypter)))
    if (encrypt)
      result.encrypt
    else
      result
  }

  // These types are used to handle errors differently during JCR-to-SQL migration.
  // When this migration is removed, it can be removed as well.

  type ErrorHandler = (=> Unit) => Unit

  type PropertyErrorHandler = String => ErrorHandler

  def defaultErrorHandler: PropertyErrorHandler = { _ =>
    block =>
      block
  }

}

@SuppressWarnings(Array("serial"))
@HttpResponseCodeResult(statusCode = 500)
class TypeNotFoundException(val message: String) extends DeployitException(message) {
}
