package com.xebialabs.deployit.security.client

import ai.digital.deploy.permissions.client.{RolePrincipalsServiceClient, RoleServiceClient => PermissionServiceRoleServiceClient}
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.sql.SqlRoleRepository
import com.xebialabs.deployit.security.{Permissions, ReadOnlyAdminRoleService, Role}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Primary
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

import java.util
import scala.jdk.CollectionConverters._

@Component
@Primary
@ConditionalOnProperty(name = Array("xl.permission-service.enabled"), havingValue = "true", matchIfMissing = true)
class RoleServiceClient(@Autowired @Qualifier("sqlRoleRepository")  sqlRoleRepository: SqlRoleRepository,
                        @Autowired val roleServiceClient: PermissionServiceRoleServiceClient,
                        @Autowired val rolePrincipalsServiceClient: RolePrincipalsServiceClient) extends ReadOnlyAdminRoleService with Logging {

  override def roleExists(roleName: String): Boolean =
    roleServiceClient.read(roleName).isDefined

  override def countRoles(onConfigurationItem: Number, rolePattern: String): Long =
    countRoles(rolePattern)

  override def countRoles(onConfigurationItem: String, rolePattern: String): Long =
    this.countRoles(rolePattern)

  private def countRoles(rolePattern: String): Long =
    roleServiceClient.countRoles(rolePattern)

  override def getRoles(rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    Option(paging).map(
      p => rolePrincipalsServiceClient.read(rolePattern,
                          p.page,
                          p.resultsPerPage,
                          getSortOrder(order),
                          "name"
      ).data
    )
    .getOrElse(
      rolePrincipalsServiceClient.readByRolePattern(rolePattern)
    ).map(dto => new Role(dto.role.id.toString, dto.role.name, dto.principals.asJavaMutable())).asJavaMutable()

  override def getRoles(onConfigurationItem: String, rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    getRoles(rolePattern, paging, order)

  override def getRolesFor(principal: String, rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    Option(paging).map(
      p => rolePrincipalsServiceClient.read(
        principal,
        rolePattern,
        p.page,
        p.resultsPerPage,
        getSortOrder(order),
        Option(order).map(_.field).getOrElse("name")
      ).data
    )
    .getOrElse(
      rolePrincipalsServiceClient.read(
        principal,
        rolePattern
      )
    )
    .map(dto => new Role(dto.role.id.toString, dto.role.name)).asJavaMutable()

  override def getRolesFor(auth: Authentication, rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] = {
    val principals = Permissions.authenticationToPrincipals(auth)
    if (principals.isEmpty) {
      new util.ArrayList()
    } else {
      Option(paging).map(p =>
        rolePrincipalsServiceClient.read(
          principals.asScala.toList,
          rolePattern,
          p.page,
          p.resultsPerPage,
          getSortOrder(order),
          "name").data
      ).getOrElse(
        rolePrincipalsServiceClient.read(
          principals.asScala.toList,
          rolePattern
        )
      ).map(dto => new Role(dto.role.id.toString, dto.role.name)).asJavaMutable()
    }
  }

  override def getRoleForRoleName(roleName: String): Role = {
    val role = roleServiceClient.read(roleName).map(dto => new Role(dto.id.toString, dto.name))
      .getOrElse(throw new NotFoundException("Could not find the role [%s]", roleName))
    role.setPrincipals(rolePrincipalsServiceClient.readPrincipals(roleName).asJavaMutable())
    role
  }

  override def getRoleForRoleId(roleId: String): Role = {
    val role = roleServiceClient.readById(roleId)
      .map(dto => new Role(dto.id.toString, dto.name))
      .getOrElse(throw new NotFoundException("Could not find the role with id [%s]", roleId))
    role.setPrincipals(rolePrincipalsServiceClient.readPrincipals(role.getName).asJavaMutable())
    role
  }

  override def readRoleAssignments(rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    getRoles(rolePattern, paging, order)

  override def readRoleAssignments(onConfigurationItem: String, rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    readRoleAssignments(rolePattern, paging, order)

  override def create(name: String, onConfigurationItem: String): String =
    sqlRoleRepository.create(name, onConfigurationItem)

  override def createOrUpdateRole(role: Role, onConfigurationItem: String): String =
    sqlRoleRepository.createOrUpdateRole(role, onConfigurationItem)

  override def rename(name: String, newName: String, onConfigurationItem: String): String =
    sqlRoleRepository.rename(name, newName, onConfigurationItem)

  override def deleteByName(name: String): Unit =
    sqlRoleRepository.deleteByName(name)

  override def deleteById(roleId: String): Unit =
    sqlRoleRepository.deleteById(roleId)

  override def writeRoleAssignments(roles: util.List[Role]): Unit =
    sqlRoleRepository.writeRoleAssignments(roles)

  override def writeRoleAssignments(onConfigurationItem: String, roles: util.List[Role]): Unit =
    sqlRoleRepository.writeRoleAssignments(roles)
}
