package com.xebialabs.deployit.security.client

import ai.digital.deploy.permissions.api.rest.dto.RoleWithPrincipalsDto
import ai.digital.deploy.permissions.client.{RolePrincipalsServiceClient, RoleServiceClient => PermissionServiceRoleServiceClient}
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.{Permissions, ReadOnlyAdminRoleService, Role}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

import java.util
import java.util.UUID
import scala.jdk.CollectionConverters._

@Component("deployRoleServiceClient")
class RoleServiceClient(@Autowired val roleServiceClient: PermissionServiceRoleServiceClient,
                        @Autowired val rolePrincipalsServiceClient: RolePrincipalsServiceClient) extends ReadOnlyAdminRoleService with Logging {

  override def roleExists(roleName: String): Boolean =
    roleServiceClient.read(roleName).isDefined

  override def countRoles(onConfigurationItem: Number, rolePattern: String): Long =
    countRoles(rolePattern)

  override def countRoles(onConfigurationItem: String, rolePattern: String): Long =
    this.countRoles(rolePattern)

  private def countRoles(rolePattern: String): Long =
    roleServiceClient.countRoles(rolePattern)

  override def getRoles(rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    Option(paging).map(
      p => rolePrincipalsServiceClient.read(rolePattern,
                          p.page,
                          p.resultsPerPage,
                          getSortOrder(order),
                          "name"
      ).data
    )
    .getOrElse(
      rolePrincipalsServiceClient.readByRolePattern(rolePattern)
    ).map(dto => new Role(dto.role.id.toString, dto.role.name, dto.principals.asJavaMutable())).asJavaMutable()

  override def getRoles(onConfigurationItem: String, rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    getRoles(rolePattern, paging, order)

  override def getRolesFor(principal: String, rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    Option(paging).map(
      p => rolePrincipalsServiceClient.read(
        principal,
        rolePattern,
        p.page,
        p.resultsPerPage,
        getSortOrder(order),
        Option(order).map(_.field).getOrElse("name")
      ).data
    )
    .getOrElse(
      rolePrincipalsServiceClient.read(
        principal,
        rolePattern
      )
    )
    .map(dto => new Role(dto.role.id.toString, dto.role.name)).asJavaMutable()

  override def getRolesFor(auth: Authentication, rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] = {
    val principals = Permissions.authenticationToPrincipals(auth)
    if (principals.isEmpty) {
      new util.ArrayList()
    } else {
      Option(paging).map(p =>
        rolePrincipalsServiceClient.read(
          principals.asScala.toList,
          rolePattern,
          p.page,
          p.resultsPerPage,
          getSortOrder(order),
          "name").data
      ).getOrElse(
        rolePrincipalsServiceClient.read(
          principals.asScala.toList,
          rolePattern
        )
      ).map(dto => new Role(dto.role.id.toString, dto.role.name)).asJavaMutable()
    }
  }

  override def getRoleForRoleName(roleName: String): Role = {
    val role = roleServiceClient.read(roleName).map(dto => new Role(dto.id.toString, dto.name))
      .getOrElse(throw new NotFoundException("Could not find the role [%s]", roleName))
    role.setPrincipals(rolePrincipalsServiceClient.readPrincipals(roleName).asJavaMutable())
    role
  }

  override def getRoleForRoleId(roleId: String): Role = {
    val role = roleServiceClient.readById(roleId)
      .map(dto => new Role(dto.role.id.toString, dto.role.name, dto.principals.asJava))
      .getOrElse(throw new NotFoundException("Could not find the role with id [%s]", roleId))
    role
  }

  override def readRoleAssignments(rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] =
    getRoles(rolePattern, paging, order)

  override def readRoleAssignments(onConfigurationItem: String, rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] = {
    if (onConfigurationItem == null) {
      readRoleAssignments(rolePattern, paging, order)
    }
    else {
      throw RoleAssignmentUnsuporrtedException("Cannot assign Configuration ID to an Role - not supported action");
    }
  }

  override def create(name: String, onConfigurationItem: String): String = {
    val roleDto = roleServiceClient.create(name)
    roleDto.id.toString
  }

  override def createOrUpdateRole(newRole: Role, onConfigurationItem: String): String = {
    val originalRole =
      Option(newRole.getId)
        .map(id =>
          roleServiceClient.readById(id)
            .getOrElse(roleServiceClient.create(newRole.getName, newRole.getPrincipals.asScala.toList))
        ).getOrElse(roleServiceClient.create(newRole.getName, newRole.getPrincipals.asScala.toList))


    val originalPrincipals = originalRole.principals.toSet
    val updatedPrincipals = newRole.getPrincipals.asScala.toSet
    val (principalsToCreate, principalsToDelete) = diff2(originalPrincipals, updatedPrincipals)
    roleServiceClient.update(originalRole.role.name, newRole.getName, principalsToCreate, principalsToDelete).role.id.toString
  }

  override def rename(name: String, newName: String, onConfigurationItem: String): String =
    roleServiceClient.rename(name, newName).id.toString

  override def deleteByName(name: String): Unit =
    roleServiceClient.removeAllReferences(name)

  override def deleteById(roleId: String): Unit = {
    val roleDto = roleServiceClient.readById(roleId)
    roleServiceClient.removeAllReferences(
      roleDto.getOrElse(throw new NotFoundException("Could not find the role for id [%s]", roleId)).role.name
    )
  }

  private def readAllRoleAssignments(): List[RoleWithPrincipalsDto] =
    rolePrincipalsServiceClient.readByRolePattern(null)

  override def writeRoleAssignments(roles: util.List[Role]): Unit =
    writeRoleAssignments(null, roles)

  override def writeRoleAssignments(onConfigurationItem: String, roles: util.List[Role]): Unit = {
    if (onConfigurationItem == null) {
      val convertedRoles = roles.asScala
      checkDuplicates(convertedRoles.toSeq)
      convertedRoles.filter(_.getId == null).foreach(generateId)

      val originalRoles = readAllRoleAssignments().map(r => new Role(r.role.id.toString, r.role.name, r.principals.asJava))

      val (toCreate, toUpdate, toDelete) = diff3(convertedRoles.toSet, originalRoles.toSet)
      toDelete.foreach(r => roleServiceClient.removeAllReferences(r.getName))
      toCreate.foreach(r => roleServiceClient.create(r.getName, r.getPrincipals.asScala.toList))
      toUpdate.foreach(r => {
        val (principalsToCreate, principalsToDelete) = diff2(r._1.getPrincipals.asScala.toSet, r._2.getPrincipals.asScala.toSet)
        roleServiceClient.update(r._1.getName, r._2.getName, principalsToCreate, principalsToDelete)
      })
    }
    else {
      throw RoleAssignmentUnsuporrtedException("Cannot assign Configuration ID to an Role - not supported action");
    }
  }

  private def generateId(role: Role): Unit = role.setId(UUID.randomUUID().toString)

  private def diff2(original: Set[String], updated: Set[String]): (Set[String], Set[String]) = (
    updated.diff(original),
    original.diff(updated)
  )

  private def diff3(updated: Set[Role], original: Set[Role]): (Set[Role], Set[(Role, Role)], Set[Role]) = (
    updated.diff(original),
    for {
      u <- updated
      o <- original
      if u == o
    } yield (o, u),
    original.diff(updated)
  )

  private def checkDuplicates(updatedRoles: Seq[Role]): Unit = {
    val duplicateRoles = updatedRoles.groupBy(_.getName).filter(_._2.size > 1)
    Checks.checkArgument(duplicateRoles.isEmpty, s"Roles with duplicate names [${duplicateRoles.keys.mkString(", ")}] are not allowed")
  }
}

case class RoleAssignmentUnsuporrtedException(message: String) extends DeployitException(message)
