package com.xebialabs.deployit.security.sql

import java.sql.ResultSet
import java.util
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{ColumnName, JoinBuilder, JoinType, OrderBy, Queries, SchemaInfo, SelectBuilder, SqlFunction, TableName, SqlCondition => cond}
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.security.model.{XldUserCredentials, XldUserProfile}
import com.xebialabs.deployit.security.repository.XldUserProfileRepository
import com.xebialabs.deployit.security.sql.SqlXldUserProfileRepository._
import com.xebialabs.deployit.security.sql.XldUserProfileSchema._
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor, RowMapper, SingleColumnRowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.util.Date
import scala.jdk.CollectionConverters._

@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
@Transactional("mainTransactionManager")
class SqlXldUserProfileRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                 (@Autowired @Qualifier("mainSchema") override implicit val schemaInfo: SchemaInfo)
  extends XldUserProfileRepository with SqlXldUserProfileRepositoryQueries {

  override def createProfile(username: String, analyticsEnabled: Boolean): Unit = {
    createUserProfile(XldUserProfile(username, analyticsEnabled))
  }

  override def updateProfile(username: String, analyticsEnabled: Boolean): Unit = {
    jdbcTemplate.update(UPDATE, analyticsEnabled, username.toLowerCase())
  }

  override def createUserProfile(profile: XldUserProfile): Unit = {
    jdbcTemplate.update(INSERT_USER_PROFILE, profile.username.toLowerCase(), profile.analyticsEnabled, profile.fullName, profile.email, profile.loginAllowed, profile.isInternal)
  }

  override def updateUserProfile(profile: XldUserProfile): Unit = {
    jdbcTemplate.update(UPDATE_USER_PROFILE, profile.analyticsEnabled, profile.fullName, profile.email, profile.loginAllowed, profile.username)
  }

  override def findOne(username: String, loadCredentials: Boolean = false): Option[XldUserProfile] = {
    val selectBuilder = new SelectBuilder(tableName)
      .where(cond.equals(USERNAME, username.toLowerCase))
      .as(userProfileTableAlias)
    if (!loadCredentials) {
      jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapRowToUserProfile)
    }.asScala.headOption else Option({
      val joinBuilder = new JoinBuilder(selectBuilder)
        .join(
          new SelectBuilder(XldUserCredentialsSchema.tableName).as(userCredentialsTableAlias),
          cond.equals(
            USERNAME.tableAlias(userProfileTableAlias),
            XldUserCredentialsSchema.PROFILE_USERNAME.tableAlias(userCredentialsTableAlias)
          ),
          JoinType.Left
        )
        .join(
          new SelectBuilder(XldUserDefaultCredentialsSchema.tableName).as(userDefaultCredentialsTableAlias),
          cond.equals(
            XldUserCredentialsSchema.ID.tableAlias(userCredentialsTableAlias),
            XldUserDefaultCredentialsSchema.USER_CREDENTIALS_ID.tableAlias(userDefaultCredentialsTableAlias)
          ),
          JoinType.Left
        )
      jdbcTemplate.query(joinBuilder.query, Setter(joinBuilder.parameters), mapRowToUserProfileWithCredentials)
    })
  }

  private def withTextContainFilter(username: String, fullName: String, email: String, builder: SelectBuilder): Unit = {
    val collectionOfCond = Seq(
      Option(username).filter(_.nonEmpty).map(un => cond.like(SqlFunction.lower(USERNAME), s"%${un.toLowerCase}%")),
      Option(fullName).filter(_.nonEmpty).map(fn => cond.like(SqlFunction.lower(FULL_NAME), s"%${fn.toLowerCase}%")),
      Option(email).filter(_.nonEmpty).map(em => cond.like(SqlFunction.lower(EMAIL), s"%${em.toLowerCase}%"))
    ).flatten

    if(collectionOfCond.nonEmpty)
      builder.where(cond.or(collectionOfCond))
  }

  override def countUsers(username: String, fullName: String, email: String): Long = {
    val builder = new SelectBuilder(tableName).select(SqlFunction.countAll)
    withTextContainFilter(username, fullName, email, builder)

    jdbcTemplate
      .query(builder.query, Setter(builder.parameters), new SingleColumnRowMapper(classOf[Long]))
      .asScala
      .headOption
      .getOrElse(0L)
  }

  override def listAllUserProfiles(username: String, fullName: String, email: String, paging: Paging, order: dto.Ordering): util.List[XldUserProfile] = {
    val selectBuilder = new SelectBuilder(tableName)

    withTextContainFilter(username, fullName, email, selectBuilder)

    if (paging != null) {
      selectBuilder.showPage(paging.page, paging.resultsPerPage)
    }

    if (order != null) {
      val columnToSort = fieldMapping(order.field)
      selectBuilder.orderBy(
        if (order.isAscending) OrderBy.asc(columnToSort) else OrderBy.desc(columnToSort)
      )
    }
    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapRowToUserProfile)
  }

  override def delete(username: String): Unit = {
    jdbcTemplate.update(DELETE, Setter(Seq(username.toLowerCase())))
  }

  override def countUserWithLoginAllowed(): Int = {
    jdbcTemplate.queryForObject(COUNT_USER_WITH_LOGIN_ALLOWED, classOf[Number], true).intValue()
  }

  override def updateLastActive(username: String, date: Date): Int = {
      jdbcTemplate.update(UPDATE_LAST_ACTIVE, date, username.toLowerCase())
  }

  private def mapRowToUserProfile: RowMapper[XldUserProfile] = (rs: ResultSet, _) =>
  XldUserProfile(
    rs.getString(USERNAME.name),
    rs.getBoolean(ANALYTICS_ENABLED.name),
    rs.getString(FULL_NAME.name),
    rs.getString(EMAIL.name),
    rs.getBoolean(LOGIN_ALLOWED.name),
    rs.getTimestamp(LAST_ACTIVE.name),
    rs.getBoolean(IS_INTERNAL.name)
  )

  private def mapRowToUserProfileWithCredentials: ResultSetExtractor[XldUserProfile] = (rs: ResultSet) => {
    var row = 0
    val list = new util.ArrayList[(String, Boolean, XldUserCredentials)]()
    while (rs.next()) {
      list.add(rowMapper.mapRow(rs, row))
      row += 1
    }
    val data = list.asScala.toList.groupBy(_._1).map {
      case (username, profileCredPairs) =>
        val first = profileCredPairs.head
        val credentials = profileCredPairs.map(_._3).toSet
        XldUserProfile(
          username,
          first._2,          // analyticsEnabled
          credentials
        )
    }.headOption.orNull
    data
  }

  private val rowMapper: RowMapper[(String, Boolean, XldUserCredentials)] = (rs: ResultSet, row: Int) => {
    (
      rs.getString(USERNAME.name),
      rs.getBoolean(ANALYTICS_ENABLED.name),
      userCredentialsRowMapper.mapRow(rs, row)
    )
  }

  private val userCredentialsRowMapper =
    SqlXldUserCredentialsRepository.createRowMapper()
}

object SqlXldUserProfileRepository {
  val userProfileTableAlias = "xup"
  val userCredentialsTableAlias = "xuc"
  val userDefaultCredentialsTableAlias = "xudc"

  private val fieldMapping: Map[String, ColumnName] = Map(
    "username" -> USERNAME,
    "fullName" -> FULL_NAME,
    "email" -> EMAIL,
    "lastActive" -> LAST_ACTIVE,
    "loginAllowed" -> LOGIN_ALLOWED,
    "isInternal" -> IS_INTERNAL
  )
}

object XldUserProfileSchema {
  val tableName: TableName = TableName("XLD_USER_PROFILES")
  val USERNAME: ColumnName = ColumnName("USERNAME")
  val ANALYTICS_ENABLED: ColumnName = ColumnName("ANALYTICS_ENABLED")
  val FULL_NAME: ColumnName = ColumnName("FULL_NAME")
  val EMAIL: ColumnName = ColumnName("USER_EMAIL")
  val LOGIN_ALLOWED: ColumnName = ColumnName("LOGIN_ALLOWED")
  val LAST_ACTIVE: ColumnName = ColumnName("LAST_ACTIVE")
  val IS_INTERNAL: ColumnName = ColumnName("IS_INTERNAL")
}

trait SqlXldUserProfileRepositoryQueries extends Queries {
  val INSERT = sqlb"insert into $tableName ($USERNAME, $ANALYTICS_ENABLED) values (?, ?)"
  val UPDATE = sqlb"update $tableName set $ANALYTICS_ENABLED = ? where $USERNAME = ?"
  val INSERT_USER_PROFILE = sqlb"insert into $tableName ($USERNAME, $ANALYTICS_ENABLED, $FULL_NAME, $EMAIL, $LOGIN_ALLOWED, $IS_INTERNAL) values (?, ?, ?, ? ,?, ?)"
  val UPDATE_USER_PROFILE = sqlb"update $tableName set $ANALYTICS_ENABLED = ? ,$FULL_NAME = ?, $EMAIL = ?, $LOGIN_ALLOWED = ? where $USERNAME = ?"
  val DELETE = sqlb"delete from $tableName where ($USERNAME) = (?)"
  val COUNT_USER_WITH_LOGIN_ALLOWED =   sqlb"SELECT COUNT(*) FROM $tableName WHERE $LOGIN_ALLOWED = ?"
  val UPDATE_LAST_ACTIVE = sqlb"update $tableName set $LAST_ACTIVE = ? where $USERNAME = ?"

}
