package com.xebialabs.deployit.security.sql

import java.sql.ResultSet
import java.util
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{ColumnName, JoinBuilder, JoinType, Queries, SchemaInfo, SelectBuilder, TableName, SqlCondition => cond}
import com.xebialabs.deployit.security.model.{XldUserCredentials, XldUserProfile}
import com.xebialabs.deployit.security.repository.XldUserProfileRepository
import com.xebialabs.deployit.security.sql.SqlXldUserProfileRepository._
import com.xebialabs.deployit.security.sql.XldUserProfileSchema._
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.util.Date
import scala.jdk.CollectionConverters._

@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
@Transactional("mainTransactionManager")
class SqlXldUserProfileRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                 (@Autowired @Qualifier("mainSchema") override implicit val schemaInfo: SchemaInfo)
  extends XldUserProfileRepository with SqlXldUserProfileRepositoryQueries {

  override def createProfile(username: String, analyticsEnabled: Boolean): Unit = {
    createUserProfile(XldUserProfile(username,analyticsEnabled))
  }

  override def updateProfile(username: String, analyticsEnabled: Boolean): Unit = {
    jdbcTemplate.update(UPDATE, analyticsEnabled, username.toLowerCase())
  }

  override def createUserProfile(profile: XldUserProfile): Unit = {
    jdbcTemplate.update(INSERT_USER_PROFILE, profile.username.toLowerCase(), profile.analyticsEnabled, profile.fullName, profile.email, profile.loginAllowed, profile.isInternal)
  }

  override def updateUserProfile(profile: XldUserProfile): Unit = {
    jdbcTemplate.update(UPDATE_USER_PROFILE, profile.analyticsEnabled, profile.fullName, profile.email, profile.loginAllowed, profile.username)
  }

  override def findOne(username: String, loadCredentials: Boolean = false): Option[XldUserProfile] = {
    val selectBuilder = new SelectBuilder(tableName)
      .where(cond.equals(USERNAME, username.toLowerCase))
      .as(userProfileTableAlias)
    if (!loadCredentials) {
      jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapRowToUserProfile)
    }.asScala.headOption else Option({
      val joinBuilder = new JoinBuilder(selectBuilder)
        .join(
          new SelectBuilder(XldUserCredentialsSchema.tableName).as(userCredentialsTableAlias),
          cond.equals(
            USERNAME.tableAlias(userProfileTableAlias),
            XldUserCredentialsSchema.PROFILE_USERNAME.tableAlias(userCredentialsTableAlias)
          ),
          JoinType.Left
        )
        .join(
          new SelectBuilder(XldUserDefaultCredentialsSchema.tableName).as(userDefaultCredentialsTableAlias),
          cond.equals(
            XldUserCredentialsSchema.ID.tableAlias(userCredentialsTableAlias),
            XldUserDefaultCredentialsSchema.USER_CREDENTIALS_ID.tableAlias(userDefaultCredentialsTableAlias)
          ),
          JoinType.Left
        )
      jdbcTemplate.query(joinBuilder.query, Setter(joinBuilder.parameters), mapRowToUserProfileWithCredentials)
    })
  }

  override def delete(username: String): Unit = {
    jdbcTemplate.update(DELETE, Setter(Seq(username.toLowerCase())))
  }

  override def countUserWithLoginAllowed(): Int = {
    jdbcTemplate.queryForObject(COUNT_USER_WITH_LOGIN_ALLOWED, classOf[Number]).intValue()
  }

  override def updateLastActive(username: String, date: Date): Int = {
      jdbcTemplate.update(UPDATE_LAST_ACTIVE, date, username.toLowerCase())
  }

  private def mapRowToUserProfile: RowMapper[XldUserProfile] = (rs: ResultSet, _) =>
  XldUserProfile(
    rs.getString(USERNAME.name),
    rs.getBoolean(ANALYTICS_ENABLED.name),
    rs.getString(FULL_NAME.name),
    rs.getString(EMAIL.name),
    rs.getBoolean(LOGIN_ALLOWED.name),
    rs.getTimestamp(LAST_ACTIVE.name),
    rs.getBoolean(IS_INTERNAL.name)
  )

  private def mapRowToUserProfileWithCredentials: ResultSetExtractor[XldUserProfile] = (rs: ResultSet) => {
    var row = 0
    val list = new util.ArrayList[(String, Boolean, XldUserCredentials)]()
    while (rs.next()) {
      list.add(rowMapper.mapRow(rs, row))
      row += 1
    }
    val data = list.asScala.toList.groupBy(_._1).map {
      case (username, profileCredPairs) =>
        val first = profileCredPairs.head
        val credentials = profileCredPairs.map(_._3).toSet
        XldUserProfile(
          username,
          first._2,          // analyticsEnabled
          credentials
        )
    }.headOption.orNull
    data
  }

  private val rowMapper: RowMapper[(String, Boolean, XldUserCredentials)] = (rs: ResultSet, row: Int) => {
    (
      rs.getString(USERNAME.name),
      rs.getBoolean(ANALYTICS_ENABLED.name),
      userCredentialsRowMapper.mapRow(rs, row)
    )
  }

  private val userCredentialsRowMapper =
    SqlXldUserCredentialsRepository.createRowMapper()
}

object SqlXldUserProfileRepository {
  val userProfileTableAlias = "xup"
  val userCredentialsTableAlias = "xuc"
  val userDefaultCredentialsTableAlias = "xudc"
}

object XldUserProfileSchema {
  val tableName: TableName = TableName("XLD_USER_PROFILES")
  val USERNAME: ColumnName = ColumnName("USERNAME")
  val ANALYTICS_ENABLED: ColumnName = ColumnName("ANALYTICS_ENABLED")
  val FULL_NAME: ColumnName = ColumnName("FULL_NAME")
  val EMAIL: ColumnName = ColumnName("USER_EMAIL")
  val LOGIN_ALLOWED: ColumnName = ColumnName("LOGIN_ALLOWED")
  val LAST_ACTIVE: ColumnName = ColumnName("LAST_ACTIVE")
  val IS_INTERNAL: ColumnName = ColumnName("IS_INTERNAL")
}

trait SqlXldUserProfileRepositoryQueries extends Queries {
  val INSERT = sqlb"insert into $tableName ($USERNAME, $ANALYTICS_ENABLED) values (?, ?)"
  val UPDATE = sqlb"update $tableName set $ANALYTICS_ENABLED = ? where $USERNAME = ?"
  val INSERT_USER_PROFILE = sqlb"insert into $tableName ($USERNAME, $ANALYTICS_ENABLED, $FULL_NAME, $EMAIL, $LOGIN_ALLOWED, $IS_INTERNAL) values (?, ?, ?, ? ,?, ?)"
  val UPDATE_USER_PROFILE = sqlb"update $tableName set $ANALYTICS_ENABLED = ? ,$FULL_NAME = ?, $EMAIL = ?, $LOGIN_ALLOWED = ? where $USERNAME = ?"
  val DELETE = sqlb"delete from $tableName where ($USERNAME) = (?)"
  val COUNT_USER_WITH_LOGIN_ALLOWED =   sqlb"SELECT COUNT(*) FROM $tableName WHERE $LOGIN_ALLOWED = true"
  val UPDATE_LAST_ACTIVE = sqlb"update $tableName set $LAST_ACTIVE = ? where $USERNAME = ?"

}
