package com.xebialabs.deployit.security.service.impl

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.deployit.security.model.XldUserProfile
import com.xebialabs.deployit.security.repository.XldUserProfileRepository
import com.xebialabs.deployit.security.service.UserProfileService
import com.xebialabs.license.{License, LicenseProperty}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import com.xebialabs.license.service.LicenseService

import java.util.Date

@Service
class UserProfileServiceImpl(@Autowired xldUserProfileRepository: XldUserProfileRepository,
                             @Autowired roleService: RoleService,
                             @Autowired licenseService: LicenseService) extends UserProfileService{
  override def findOne(username: String, loadCredentials: Boolean): XldUserProfile = {
    xldUserProfileRepository.findOne(username, loadCredentials)
      .getOrElse(throw new NotFoundException(s"No user profile found for user: $username"))
  }

  override def findOne(username: String): XldUserProfile = findOne(username, loadCredentials = false)

  override def createOrUpdate(username: String, analyticsEnabled: Boolean): Unit = {
    xldUserProfileRepository.findOne(username) match {
      case Some(_) => xldUserProfileRepository.updateProfile(username, analyticsEnabled)
      case None => xldUserProfileRepository.createProfile(username, analyticsEnabled)
    }
  }

  override def createOrUpdateUserProfile(profile: XldUserProfile): Unit = {
    val updatedProfile = revokeLoginAllowedAccordingTo(profile, licenseService.getLicense)
    xldUserProfileRepository.findOne(updatedProfile.username) match {
      case Some(_) => xldUserProfileRepository.updateUserProfile(updatedProfile)
      case None => xldUserProfileRepository.createUserProfile(updatedProfile)
    }
  }

  private def revokeLoginAllowedAccordingTo(profile: XldUserProfile, license: License): XldUserProfile = {
    if (!profile.loginAllowed)
      return profile
    if (license.getStringValue(LicenseProperty.MAX_NUMBER_OF_USERS) != null) {
      val maxNumberOfUsers = Integer.parseInt(license.getStringValue(LicenseProperty.MAX_NUMBER_OF_USERS))
      if (maxNumberOfUsers != null && maxNumberOfUsers <= countUserWithLoginAllowed())
       return XldUserProfile(profile.username, profile.analyticsEnabled, profile.fullName, profile.email, false, profile.lastActive, profile.isInternal)
    }
    profile
  }

  override def countUserWithLoginAllowed(): Int = xldUserProfileRepository.countUserWithLoginAllowed()

  override def updateLastActive(username: String, date: Date): Int = xldUserProfileRepository.updateLastActive(username,date)


}