package com.xebialabs.deployit.engine.tasker.repository.sql

import ai.digital.deploy.task.serdes.TaskAkkaSerializer
import java.lang.Boolean.TRUE
import java.sql.{Connection, ResultSet}
import java.util

import ai.digital.configuration.central.deploy.TaskerSystemProperties
import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.spring.Setter.setString
import com.xebialabs.deployit.core.util.TypeConversions.enforceBoolean
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock
import com.xebialabs.deployit.engine.tasker.repository.sql.PendingTaskSchema._
import com.xebialabs.deployit.engine.tasker.repository.sql.{PendingTaskMetadataSchema => md}
import com.xebialabs.deployit.engine.tasker.repository.{PendingTask, PendingTaskRepository}
import com.xebialabs.deployit.engine.tasker.{TaskId, TaskNotFoundException, TaskSpecification}
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
@Transactional("mainTransactionManager")
class SqlPendingTaskRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                               @Autowired taskerSystemProperties: TaskerSystemProperties)
                              (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends PendingTaskRepository with PendingTasksQueries with PendingTaskMetadataQueries with Logging {

  override def tasks(loadFullSpec: Boolean): util.List[TaskWithBlock] = {
    val tasks: List[TaskWithBlock] = if (loadFullSpec) {
      jdbcTemplate.query(SELECT_ALL_SPECS, (rs: ResultSet, _: Int) => toPendingTask(rs)).asScala.toList
    } else {
      jdbcTemplate.queryForList(SELECT_ALL).asScala
        .groupBy(_.get(task_id.name).asInstanceOf[String])
        .map { case (_, list) =>
          val task = list.headOption.getOrElse(throw new NoSuchElementException("No pending task"))
          val taskId = task.get(task_id.name).asInstanceOf[String]
          val taskDescription = task.get(task_description.name).asInstanceOf[String]
          val owner = task.get(task_owner.name).asInstanceOf[String]
          val workerAddress = task.get(worker_address.name).asInstanceOf[String]
          val scheduledDate = Option(task.get(scheduled_date.name)).map(mapDateTime).orNull
          val isSentToQueue = enforceBoolean(task.get(is_sent_to_queue.name))
          val queuedDate = Option(task.get(queued_date.name)).map(mapDateTime).orNull
          val metadata = list.map { item =>
            val key = item.get(md.metadata_key.name).asInstanceOf[String]
            val value = item.get(md.metadata_value.name).asInstanceOf[String]
            key -> value
          }.toMap.asJava
          new PendingTask(taskId, taskDescription, owner, metadata, scheduledDate, None, Option(workerAddress), queuedDate, isSentToQueue)
        }.toList
    }
    tasks.asJava
  }

  override def scheduledTasks(): util.List[TaskWithBlock] =
    jdbcTemplate.query(SELECT_SCHEDULED, new RowMapper[TaskWithBlock] {
      override def mapRow(rs: ResultSet, rowNum: Int): TaskWithBlock = toPendingTask(rs)
    })

  override def queuedTasks(): util.List[PendingTask] =
    jdbcTemplate.query(SELECT_QUEUED, new RowMapper[PendingTask] {
      override def mapRow(rs: ResultSet, rowNum: Int): PendingTask = toPendingTask(rs)
    })

  override def task(taskId: TaskId, loadFullSpec: Boolean = false): Option[PendingTask] = {
    if (loadFullSpec) readFullSpec(taskId) else readSummary(taskId)
  }

  private def readSummary(taskId: TaskId): Option[PendingTask] = {
    logger.debug(s"Reading the summary of task [$taskId]")
    try {
      firstOrNone(jdbcTemplate.query(SELECT, (rs: ResultSet, _: Int) => {
        val description = rs.getString(task_description.name)
        val owner = rs.getString(task_owner.name)
        val workerAddress = extractWorkerAddress(rs)
        val scheduledDate = extractScheduledDate(rs)
        val isSentToQueue = extractIsSentToQueue(rs)
        val queuedDate = extractQueuedDate(rs)
        (description, owner, scheduledDate, workerAddress, queuedDate, isSentToQueue)
      }, taskId)).map { case (taskDescription, owner, scheduledDate, workerAddress, queuedDate, isSentToQueue) =>
        val metadata = jdbcTemplate.query(SELECT_METADATA, (rs: ResultSet) => {
          val map = new util.HashMap[String, String]()
          while (rs.next()) {
            map.put(rs.getString(md.metadata_key.name), rs.getString(md.metadata_value.name))
          }
          map
        }, taskId)
        new PendingTask(taskId, taskDescription, owner, metadata, scheduledDate, None, workerAddress, queuedDate, isSentToQueue)
      }
    } catch {
      case e: Exception =>
        logger.error(e.getMessage, e)
        None
    }
  }

  private def readFullSpec(taskId: TaskId): Option[PendingTask] = {
    logger.debug(s"Reading the full specification of task [$taskId]")
    try {
      firstOrNone(jdbcTemplate.query(SELECT_SPEC, (rs: ResultSet, _: Int) => toPendingTask(rs), taskId))
    } catch {
      case e: Exception =>
        logger.error(e.getMessage, e)
        None
    }
  }

  private def toPendingTask(rs: ResultSet): PendingTask =
    pendingTaskFromBinary(extractTaskId(rs), extractTaskSpecification(rs), extractScheduledDate(rs), extractWorkerAddress(rs), extractQueuedDate(rs), extractIsSentToQueue(rs))

  private def pendingTaskFromBinary(taskId: String, taskSpecBytes: Array[Byte], scheduledDate: DateTime,
                                    workerAddress: Option[String], queuedDate: DateTime, isSentToQueue: Boolean = false): PendingTask = {
    serializer.fromBinary(taskSpecBytes, classOf[TaskSpecification]) match {
      case spec: TaskSpecification =>
        val pt = new PendingTask(taskId, spec.getDescription, spec.getOwner.getName, spec.getMetadata, scheduledDate, Some(spec), workerAddress, queuedDate, isSentToQueue)
        logger.debug(s"Pending task from binary has been created $pt")
        pt
      case _ =>
        throw new RuntimeException(s"Could not deserialize TaskSpecification [$taskId]")
    }
  }

  def store(spec: TaskSpecification, workerAddress: Option[String]): TaskId = {
    val taskId = spec.getId
    val bytes = specToBinary(spec)

    jdbcTemplate.update(
      INSERT,
      taskId,
      spec.getDescription,
      spec.getOwner.getName,
      bytes,
      workerAddress.orNull
    )
    spec.getMetadata.forEach { (k, v) =>
      jdbcTemplate.update(INSERT_METADATA, taskId, k, v)
    }
    taskId
  }

  override def update(taskId: TaskId, spec: TaskSpecification): Unit = {
    val bytes = specToBinary(spec)
    try {
      jdbcTemplate.update(
        UPDATE,
        bytes,
        taskId
      )
    } catch {
      case _: Exception => throw new TaskNotFoundException("pending tasks", taskId)
    }
  }

  private def specToBinary(spec: TaskSpecification): Array[Byte] = {
    Try(serializer.toBinary(spec)) match {
      case Success(value) => value
      case Failure(exception) =>
        logger.error(exception.getMessage, exception)
        throw exception
    }
  }

  private lazy val serializer = {
    if(taskerSystemProperties.serialization.`type`.equals("jackson-yaml")) {
      TaskAkkaSerializer.getTaskAkkaSerializer(classOf[ai.digital.deploy.task.serdes.yaml.TaskerSerializableYaml])
    } else {
      TaskAkkaSerializer.getTaskAkkaSerializer(classOf[ai.digital.deploy.task.serdes.kryo.TaskerSerializableKryo])
    }
  }

  override def changeOwner(taskId: TaskId, newOwner: String): Unit = {
    jdbcTemplate.update(UPDATE_OWNER, newOwner, taskId)
  }

  override def markAsSentToQueue(taskId: String): Unit = {
    logger.debug(s"Marking task [$taskId] as sent to the queue")
    jdbcTemplate.update({ con: Connection =>
      val ps = con.prepareStatement(UPDATE_SENT_TO_QUEUE_AND_QUEUED_DATE)
      ps.setBoolean(1, TRUE)
      setTimestamp(ps, 2, DateTime.now())
      setString(ps, 3, taskId)
      ps
    })
  }

  override def schedule(taskId: TaskId, date: DateTime): Unit = {
    logger.debug(s"Task [$taskId] has been scheduled")
    jdbcTemplate.update(UPDATE_SCHEDULED_DATE, toTimestamp(date), taskId)
  }

  override def delete(taskId: TaskId): Unit = {
    jdbcTemplate.update(DELETE_METADATA, taskId)
    jdbcTemplate.update(DELETE, taskId)
    logger.debug(s"Pending task $taskId has been deleted from database")
  }

  override def prepareToEnqueue(taskId: TaskId): Try[TaskWithBlock] = {
    logger.debug(s"Preparing task [taskId] to enqueue")

    val q = schemaInfo.sqlDialect.lockSelectBuilder(tableName)
      .select(task_id).select(scheduled_date).select(task_specification).select(worker_address).select(queued_date).select(is_sent_to_queue)
      .where(SqlCondition.equals(task_id, taskId))
    firstOrNone[TaskWithBlock](jdbcTemplate.query(q.query, (rs: ResultSet, _: Int) => {
      val scheduledDate = extractScheduledDate(rs)
      val workerAddress = extractWorkerAddress(rs)
      val queuedDate = extractQueuedDate(rs)
      pendingTaskFromBinary(extractTaskId(rs), extractTaskSpecification(rs), scheduledDate, workerAddress, queuedDate, extractIsSentToQueue(rs))
    }, taskId)) match {
      case None =>
        Failure(new RuntimeException(s"Task [$taskId] was not found"))
      case Some(task: PendingTask) if task.isSentToQueue =>
        Failure(new RuntimeException(s"Task [$taskId] is already sent to queue"))
      case Some(task) => Success(task)
    }
  }

  private def extractTaskId(rs: ResultSet): TaskId = rs.getString(task_id.name)

  private def extractWorkerAddress(rs: ResultSet): Option[String] = Option(rs.getString(worker_address.name))

  private def extractScheduledDate(rs: ResultSet): DateTime = Option(rs.getTimestamp(scheduled_date.name)).map(t => toDateTime(t)).orNull

  private def extractTaskSpecification(rs: ResultSet) = rs.getBytes(task_specification.name)

  private def extractIsSentToQueue(rs: ResultSet): Boolean = rs.getBoolean(is_sent_to_queue.name)

  private def extractQueuedDate(rs: ResultSet): DateTime = Option(rs.getTimestamp(queued_date.name)).map(t => toDateTime(t)).orNull

}

object PendingTaskSchema {
  val tableName: TableName = TableName("XLD_PENDING_TASKS")

  val task_id: ColumnName = ColumnName("task_id")
  val task_description: ColumnName = ColumnName("description")
  val task_owner: ColumnName = ColumnName("task_owner")
  val scheduled_date: ColumnName = ColumnName("scheduled_date")
  val worker_address: ColumnName = ColumnName("worker_address")
  val task_specification: ColumnName = ColumnName("task_specification")
  val is_sent_to_queue: ColumnName = ColumnName("is_sent_to_queue")
  val queued_date: ColumnName = ColumnName("queued_date")
}

object PendingTaskMetadataSchema {
  val tableName: TableName = TableName("XLD_PENDING_TASKS_METADATA")

  val task_id: ColumnName = ColumnName("task_id")
  val metadata_key: ColumnName = ColumnName("metadata_key")
  val metadata_value: ColumnName = ColumnName("metadata_value")
}

trait PendingTasksQueries extends Queries {

  import PendingTaskSchema._

  lazy val INSERT = sqlb"insert into $tableName ($task_id, $task_description, $task_owner, $task_specification, $worker_address) values (?,?,?,?, ?)"
  lazy val UPDATE = sqlb"update $tableName set $task_specification = ? where $task_id = ?"
  lazy val UPDATE_OWNER = sqlb"update $tableName set $task_owner = ? where $task_id = ?"
  lazy val SELECT = sqlb"select $task_id, $task_description, $task_owner, $scheduled_date, $worker_address, $queued_date, $is_sent_to_queue from $tableName where $task_id = ?"
  lazy val SELECT_ALL: TaskId = sqlb"select t.$task_id, t.$task_description, t.$task_owner, t.$scheduled_date, t.$worker_address, t.$queued_date, t.$is_sent_to_queue, " +
    sqlb"md.${md.metadata_key}, md.${md.metadata_value} from $tableName t left join ${md.tableName} md on t.$task_id = md.${md.task_id}"

  lazy val SELECT_ALL_SPECS = sqlb"select $task_id, $scheduled_date, $task_specification, $worker_address, $queued_date, $is_sent_to_queue from $tableName"
  lazy val SELECT_SPEC = sqlb"select $task_id, $scheduled_date, $task_specification, $worker_address, $queued_date, $is_sent_to_queue from $tableName where $task_id = ?"

  lazy val DELETE = sqlb"delete from $tableName where $task_id = ?"
  lazy val UPDATE_SCHEDULED_DATE = sqlb"update $tableName set $scheduled_date = ? where $task_id = ?"
  lazy val UPDATE_SENT_TO_QUEUE_AND_QUEUED_DATE = sqlb"update $tableName set $is_sent_to_queue = ?, $queued_date = ? where $task_id = ?"
  lazy val SELECT_SCHEDULED = sqlb"select $task_id, $scheduled_date, $task_specification, $worker_address, $queued_date, $is_sent_to_queue from $tableName where $scheduled_date is not null"
  lazy val SELECT_QUEUED = sqlb"select $task_id, $scheduled_date, $task_specification, $worker_address, $queued_date, $is_sent_to_queue from $tableName where $queued_date is not null order by $queued_date"
}


trait PendingTaskMetadataQueries extends Queries {

  import PendingTaskMetadataSchema._

  lazy val INSERT_METADATA = sqlb"insert into $tableName (${md.task_id}, $metadata_key, $metadata_value) values (?,?,?)"
  lazy val SELECT_METADATA = sqlb"select $metadata_key, $metadata_value from $tableName where ${md.task_id} = ?"
  lazy val DELETE_METADATA = sqlb"delete from $tableName where ${md.task_id} = ?"
}
