package com.xebialabs.deployit.engine.tasker.repository.sql

import java.sql.ResultSet

import com.xebialabs.deployit.core.events.TaskPathStatusEvent
import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs, BatchExecutorRepository}
import com.xebialabs.deployit.engine.api.dto.TaskPathStatus
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.engine.tasker.repository.TaskPathStatusRepository
import com.xebialabs.deployit.sql.base.schema.ActiveTaskPathStatusSchema._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

@Component
class SqlActiveTaskPathStatusRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                        @Autowired @Qualifier("mainBatchExecutorRepository") val batchExecutorRepository: BatchExecutorRepository)
                                       (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends TaskPathStatusRepository with ActiveTaskPathStatusQueries with Logging {

  private val SELECT_TASK_STATUS_BY_PATH_WITH_LIMIT =
    schemaInfo.sqlDialect.addPaging(SELECT_TASK_STATUS_BY_PATH, 1, 1)

  @Transactional(value = "mainTransactionManager", readOnly = true)
  override def findOne(taskId: TaskId, path: String): Option[TaskPathStatus] = {
    firstOrNone(jdbcTemplate.query(SELECT_TASK_STATUS_BY_PATH_WITH_LIMIT, mapToRow, taskId, path))
  }

  @Transactional(value = "mainTransactionManager")
  override def create(taskId: TaskId, path: String, status: String, updateTime: Long): Unit =
    jdbcTemplate.update(INSERT_TASK_STATUS, taskId, path, status, updateTime)

  @Transactional(value = "mainTransactionManager")
  override def batchCreate(taskPathStatuses: Seq[TaskPathStatusEvent], updateTime: Long): Unit = {
    val commands: Seq[BatchCommandWithArgs] = taskPathStatuses.map(taskPathStatus =>
      BatchCommand(INSERT_TASK_STATUS, taskPathStatus.taskId, taskPathStatus.path, taskPathStatus.status, updateTime)
    )
    batchExecutorRepository.execute(commands)
  }

  @Transactional("mainTransactionManager")
  override def delete(taskId: TaskId): Unit =
    jdbcTemplate.update(DELETE_TASK_STATUS, taskId)

  private def mapToRow: RowMapper[TaskPathStatus] =
    (rs: ResultSet, _) => TaskPathStatus(rs.getString(taskId.name), rs.getString(path.name), rs.getString(status.name))
}

private[sql] trait ActiveTaskPathStatusQueries extends Queries {

  private[sql] lazy val INSERT_TASK_STATUS: String = sqlb"insert into $tableName ($taskId, $path, $status, $updateTime) values (?,?,?,?)"
  private[sql] lazy val SELECT_TASK_STATUS_BY_PATH: String = sqlb"select $taskId, $path, $status from $tableName where $taskId = ? and $path = ? ORDER BY $updateTime DESC, $ID DESC"
  private[sql] lazy val DELETE_TASK_STATUS: String = sqlb"delete from $tableName where $taskId = ?"

}
