package ai.digital.deploy.task.steplog.queue


import ai.digital.deploy.engine.tasker.service.ActiveTaskStepLogService
import com.xebialabs.deployit.core.events.{TaskStepLogDeleteEvent, TaskStepLogEvent}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.jms.annotation.JmsListener
import org.springframework.jms.core.JmsTemplate
import org.springframework.jms.support.converter.MessageConverter
import org.springframework.stereotype.Component

import jakarta.jms.TextMessage

@ConditionalOnProperty(prefix = "deploy.task", name = Array("resilient"), havingValue = "true", matchIfMissing = false)
@Component
class TaskStepLogConsumer(taskStepLogService: ActiveTaskStepLogService,
                          jacksonJmsMessageConverter: MessageConverter,
                          @Qualifier("taskStepLogJmsTemplate") jmsTemplate: JmsTemplate
                         ) extends Logging {

  @JmsListener(destination = TaskStepLogQueue.TaskStepLogQueueELExpression,
    containerFactory = "xlJmsListenerContainerFactory")
  def receiveTextMessage(m: TextMessage): Unit =
    jacksonJmsMessageConverter.fromMessage(m) match {
      case taskStepLogEventMessage: TaskStepLogEventMessage =>
        logger.debug(
          s"Received taskStepLogEvent ${m.getJMSMessageID} timestamp: ${m.getJMSTimestamp} size: ${taskStepLogEventMessage.events.size}"
        )
        taskStepLogService.batchCreate(taskStepLogEventMessage.events, m.getJMSTimestamp)

      case taskStepLogEvent: TaskStepLogEvent =>
        logger.debug(s"Received message ${m.getJMSMessageID} timestamp: ${m.getJMSTimestamp} " +
          s"taskId: ${taskStepLogEvent.taskId} " +
          s"path: ${taskStepLogEvent.stepPath} " +
          s"logLine: ${taskStepLogEvent.lineNumber} " +
          s"logLevel: ${taskStepLogEvent.logLevel} " +
          s"log: ${taskStepLogEvent.log}")
        taskStepLogService.create(
          taskStepLogEvent, m.getJMSTimestamp)

      case taskStepLogDeleteEvent: TaskStepLogDeleteEvent =>
        logger.debug(s"Received taskStepLogDeleteEvent ${m.getJMSMessageID} " +
          s"timestamp: ${m.getJMSTimestamp} taskId: ${taskStepLogDeleteEvent.taskId}")
        taskStepLogService.delete(taskStepLogDeleteEvent.taskId)

      case _ =>
        logger.error(s"Received unknown ${m.getJMSMessageID} " +
          s"timestamp: ${m.getJMSTimestamp} [$m]")
    }

}

object TaskStepLogQueue {
  final val TaskStepLogQueueELExpression = "#{@taskStepLogQueueNameResolver.getTaskStepLogQueueName()}"
}
