package com.xebialabs.deployit.core.rest.json;

import com.google.common.annotations.VisibleForTesting;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xltype.serialization.json.JsonWriter;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.ws.rs.Consumes;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.Provider;
import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Set;

@Component
@Provider
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
public class DescriptorJsonWriter implements MessageBodyWriter<Descriptor> {

    @Override
    public boolean isWriteable(Class<?> type, final java.lang.reflect.Type genericType, final Annotation[] annotations, final MediaType mediaType) {
        return Descriptor.class.isAssignableFrom(type);
    }

    @Override
    public long getSize(final Descriptor descriptor, final Class<?> type, final java.lang.reflect.Type genericType, final Annotation[] annotations, final MediaType mediaType) {
        return -1L;
    }

    @Override
    public void writeTo(final Descriptor descriptor, final Class<?> type, final java.lang.reflect.Type genericType, final Annotation[] annotations, final MediaType mediaType, final MultivaluedMap<String, Object> httpHeaders, final OutputStream entityStream) throws IOException, WebApplicationException {
        entityStream.write(toJson(descriptor).getBytes());
    }

    @VisibleForTesting
    String toJson(Descriptor descriptor) {
        StringWriter stringWriter = new StringWriter();
        JsonWriter writer = new JsonWriter(stringWriter);
        writeJson(descriptor, writer);
        return stringWriter.toString();
    }

    public void writeJson(final Descriptor descriptor, final JsonWriter writer) {
        writer.object();
        writer.key("type").value(descriptor.getType().toString());
        if (descriptor.getDeployableType() != null) {
            writer.key("deployableType").value(descriptor.getDeployableType().toString());
        }
        if (descriptor.getContainerType() != null) {
            writer.key("containerType").value(descriptor.getContainerType().toString());
        }
        writer.key("virtual").value(descriptor.isVirtual());
        if (descriptor.getIcon() != null) {
            writer.key("icon").value(descriptor.getIcon());
        }
        if (descriptor.getRootName() != null && !descriptor.getRootName().isEmpty()) {
            writer.key("root").value(descriptor.getRootName());
        }
        writer.key("description").value(descriptor.getDescription());

        writePropertyDescriptors(writer, descriptor);

        writeIntefaces(writer, descriptor);

        writeSuperClasses(writer, descriptor);

        writeControlTasks(descriptor, writer);

        writer.endObject();
    }

    private void writeControlTasks(Descriptor descriptor, JsonWriter writer) {
        if (!CollectionUtils.isEmpty(descriptor.getControlTasks())) {
            writer.key("control-tasks");
            writer.array();
            for (MethodDescriptor controlTask : descriptor.getControlTasks()) {
                MethodDescriptorJsonWriter.writeJson(controlTask, writer);
            }
            writer.endArray();
        }
    }

    private void writeSuperClasses(JsonWriter writer, Descriptor descriptor) {
        List<Type> superTypes = descriptor.getSuperClasses();
        writer.key("superTypes");
        writer.array();
        for (Type type : superTypes) {
            writer.value(type.toString());
        }
        writer.endArray();
    }

    private void writeIntefaces(JsonWriter writer, Descriptor descriptor) {
        Set<Type> interfaces = descriptor.getInterfaces();
        writer.key("interfaces");
        writer.array();
        for (Type type : interfaces) {
            writer.value(type.toString());
        }
        writer.endArray();
    }

    private static void writePropertyDescriptors(JsonWriter writer, Descriptor descriptor) {
        writer.key("properties");
        writer.array();

        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            PropertyDescriptorJsonWriter.writePropertyDescriptor(writer, pd);
        }

        writer.endArray();
    }
}
