package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.deployment.ChangeSetBuilder;
import com.xebialabs.deployit.deployment.planner.ContainersToEnvironmentContributor;
import com.xebialabs.deployit.deployment.planner.MultiDeltaSpecification;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.repository.ChangeSet;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.deployit.service.dependency.DependencyService;
import com.xebialabs.deployit.service.deployment.DeploymentService;
import com.xebialabs.deployit.service.deployment.TaskSpecificationService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Maps.uniqueIndex;
import static com.xebialabs.deployit.core.rest.api.DeploymentWriter.createDeployment;
import static com.xebialabs.deployit.engine.api.dto.Deployment.DeploymentType.UNDEPLOYMENT;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.UNDEPLOY;
import static com.xebialabs.xlplatform.utils.Implicits.listOfListToJavaListOfList;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;

@Service("taskSpecificationService")
public class TaskSpecificationServiceImpl extends AbstractSecuredResource implements TaskSpecificationService {

    protected static final Logger logger = LoggerFactory.getLogger(TaskSpecificationServiceImpl.class);

    private final RepositoryService repositoryService;
    private final DeploymentService deploymentService;
    private final DependencyService dependencyService;
    private final WorkDirFactory workDirFactory;

    @Autowired
    public TaskSpecificationServiceImpl(RepositoryService repositoryService,
                                        DeploymentService deploymentService,
                                        DependencyService dependencyService,
                                        WorkDirFactory workDirFactory) {
        this.repositoryService = repositoryService;
        this.deploymentService = deploymentService;
        this.dependencyService = dependencyService;
        this.workDirFactory = workDirFactory;
    }

    @Override
    public TaskSpecification createDeploymentTask(Deployment deployment, boolean previewMode) {
        switch (deployment.getDeploymentType()) {
            case INITIAL:
            case UPDATE:
                return createTask(deployment, previewMode);
            case UNDEPLOYMENT:
                return createUndeploymentTask(deployment, previewMode);
            default:
                throw new IllegalArgumentException("Unknown deployment type: " + deployment.getDeploymentType());
        }
    }

    private TaskSpecification createTask(Deployment deployment, boolean previewMode) {
        WorkDir workDirForExistingDeployment = workDirFactory.newWorkDir("previous");
        logger.trace("Got workdir [{}] for reading existing deployment", workDirForExistingDeployment);
        try {
            MultiDeltaSpecification deltaSpecification =
                    createDeltaSpecification(deployment,
                            readExistingDeployedApplication(deployment, workDirForExistingDeployment));
            if (!previewMode) {
                repositoryService.checkReferentialIntegrity(ChangeSetBuilder.determineChanges(deltaSpecification));
            }
            return deploymentService.getTaskFullSpecification(deployment.getId(), deltaSpecification, WorkdirHolder.get(), workDirForExistingDeployment);
        } catch (RuntimeException e) {
            workDirForExistingDeployment.delete();
            WorkdirHolder.get().delete();
            throw e;
        } finally {
            if (previewMode) {
                workDirForExistingDeployment.delete();
            }
        }
    }

    private MultiDeltaSpecification createDeltaSpecification(Deployment deployment, final Map<String, DeployedApplication> existingDeployedApplications) {
        switch (deployment.getDeploymentType()) {
            case INITIAL:
                return deploymentService.prepareInitialSpecification(deployment, existingDeployedApplications);
            case UPDATE:
                return deploymentService.prepareUpgradeSpecification(deployment, existingDeployedApplications);
            default:
                throw new IllegalArgumentException("Deployment " + deployment.getId() + " has invalid type " + deployment.getDeploymentType());
        }

    }

    private TaskSpecification createUndeploymentTask(Deployment deployment, boolean previewMode) {
        DeployedApplication deployedApplication = (DeployedApplication) deployment.getDeployedApplication();
        checkPermission(UNDEPLOY, deployedApplication.getEnvironment().getId());
        if (deployedApplication.isUndeployDependencies() && deployment.getGroupedRequiredDeployments().isEmpty()) {
            applyUnDeploymentGroups(deployedApplication, deployment);
        }
        MultiDeltaSpecification deltaSpecification = deploymentService.prepareUndeploymentWithDependencies(deployment);
        if (!previewMode) {
            final ChangeSet changeSet = ChangeSetBuilder.determineChanges(deltaSpecification);
            new ContainersToEnvironmentContributor().contribute(deltaSpecification, changeSet, null);
            repositoryService.checkReferentialIntegrity(changeSet);
        }

        return deploymentService.getTaskFullSpecification(deployment.getId(), deltaSpecification, WorkdirHolder.get());
    }

    private Map<String, DeployedApplication> readExistingDeployedApplication(Deployment deployment,
                                                                             final WorkDir workDir) {
        return uniqueIndex(
                transform(filter(Iterables.concat(singleton(deployment), deployment.getRequiredDeployments()),
                        input -> input.isOfType(Deployment.DeploymentType.UPDATE)),
                        (Function<Deployment, DeployedApplication>) input ->
                                repositoryService.read(input.getDeployedApplication().getId(), workDir)
                ),
                BaseConfigurationItem::getId
        );
    }

    @Override
    public void applyUnDeploymentGroups(DeployedApplication deployedApplication,
                                        Deployment unDeployment) {
        if (ServerConfiguration.getInstance().isServerResolveApplicationDependencies()) {
            List<List<DeployedApplication>> dependenciesToDelete =
                    listOfListToJavaListOfList(dependencyService.findDependencyGroupsToDeleteAndValidate(deployedApplication));
            List<List<Deployment>> unDeploymentGroups =
                    dependenciesToDelete.stream().map(group -> group.stream().map(app -> createDeployment(app, UNDEPLOYMENT))
                            .collect(toList())).collect(toList());
            unDeployment.setGroupedRequiredDeployments(unDeploymentGroups);
        }
    }

}
