package com.xebialabs.deployit.core.rest.api;

import ai.digital.deploy.sql.http.enricher.PaginationService;
import com.xebialabs.deployit.core.rest.api.support.TaskPermissionFilter;
import com.xebialabs.deployit.engine.api.TaskBlockService;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.api.execution.*;
import com.xebialabs.deployit.engine.api.task.TaskBlockServiceDefaults;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.tasker.Block;
import com.xebialabs.deployit.engine.tasker.BlockPath;
import com.xebialabs.deployit.engine.tasker.Task;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.TaskMetadata;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import org.jboss.resteasy.spi.HttpResponse;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import scala.Option;

import javax.ws.rs.core.Context;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.TASK_SKIPSTEP;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;

@Service
public class TaskBlockResource extends AbstractTaskResource implements TaskBlockService, TaskBlockServiceDefaults {

    private static final Function<ArchivedTask, TaskWithBlock> castToTaskWithBlock = input -> (TaskWithBlock) input;

    @Context
    private HttpResponse response;

    @Autowired
    private RoleService roleService;

    @Override
    public List<TaskWithBlock> getMyCurrentTasks(FetchMode fetchMode) {
        final String principal = Permissions.getAuthenticatedUserName();
        return getAllTasks(fetchMode).filter(input -> input.getOwner().equals(principal)).collect(Collectors.toList());
    }

    @Override
    public List<TaskWithBlock> getAllCurrentTasks(FetchMode fetchMode) {
        Stream<TaskWithBlock> tasks = getAllTasks(fetchMode);
        if (hasPermission(ADMIN)) {
            logger.debug("getAllCurrentTasks returning full list as ADMIN");
            return tasks.collect(Collectors.toList());
        }
        return tasks.filter(new TaskPermissionFilter()).collect(Collectors.toList());
    }

    private Stream<TaskWithBlock> getAllTasks(FetchMode fetchMode) {
        return engine.get().getAllIncompleteTasks(fetchMode).stream();
    }

    private boolean filterBySatelliteId(TaskWithBlock task, String satelliteId) {
        String idsAsString = task.getMetadata().get(TaskMetadata.SATELLITE_IDS);
        if (idsAsString != null) {
            Set<String> satelliteIds = new HashSet<>(Arrays.asList(idsAsString.split(",")));
            return satelliteIds.contains(satelliteId);
        }
        return false;
    }

    @Override
    public List<TaskWithBlock> getMySatelliteTasks(String satelliteId, FetchMode fetchMode) {
        final String principal = Permissions.getAuthenticatedUserName();
        return engine.get().getAllIncompleteTasks(fetchMode).stream()
                .filter(input -> filterBySatelliteId(input, satelliteId))
                .filter(input -> input.getOwner().equals(principal))
                .collect(Collectors.toList());
    }

    @Override
    public List<TaskWithBlock> getCurrentSatelliteTasks(String satelliteId, FetchMode fetchMode) {
        List<TaskWithBlock> tasks = engine.get().getAllIncompleteTasks(fetchMode).stream()
                .filter(input -> filterBySatelliteId(input, satelliteId))
                .collect(Collectors.toList());

        if (hasPermission(ADMIN) || roleService.isReadOnlyAdmin()) {
            logger.debug("getAllCurrentTasks returning full list as ADMIN");
            return new ArrayList<>(tasks);
        }
        return tasks.stream().filter(new TaskPermissionFilter()).collect(Collectors.toList());
    }

    @Override
    public TaskWithBlock getTask(final String taskId) {
        return (TaskWithBlock) viewAbleTask(pendingOrLiveOrArchivedTask(taskId, false));
    }

    @Override
    public BlockState getBlock(final String taskId, final String blockId) {
        TaskWithBlock taskWithBlock = (TaskWithBlock) viewAbleTask(pendingOrLiveOrArchivedTask(taskId, false));
        final Option<Block> block = ((Block) taskWithBlock.getBlock()).getBlock(BlockPath.apply(blockId).tail());
        return (BlockState) block.map(b -> stepLogRetriever.retrieveLogs(taskId, b)).getOrElse(null);
    }

    @Override
    public StepBlockState getSteps(final String taskId, final String blockId) {
        TaskWithBlock task = (TaskWithBlock) viewAbleTask(pendingOrLiveOrArchivedTask(taskId, true));
        final Option<Block> blockOption = ((Block) task.getBlock()).getBlock(BlockPath.apply(blockId).tail());
        final Block block = blockOption.getOrElse(null);
        if (block instanceof StepBlockState) {
            return new StepBlockStateView((StepBlockState) block);
        } else {
            throw new DeployitException("Block is a composite block so it has no steps");
        }
    }

    @Override
    public TaskWithBlock skip(final String taskId, final List<String> stepPaths) {
        return skip(taskId, stepPaths, true);
    }

    @Override
    public TaskWithBlock unskip(final String taskId, final List<String> stepPaths) {
        return skip(taskId, stepPaths, false);
    }

    @Override
    public StepBlockState addPause(final String taskId, final String stepPath) {
        pause(taskId, stepPath);
        Task task = engine.get().retrieve(taskId);
        final Option<Block> block = task.getBlock(BlockPath.apply(stepPath).init());
        return new StepBlockStateView(block.getOrElse(null));
    }

    @Override
    public TaskWithBlock assign(final String taskId, final String owner) {
        return doAssign(taskId, owner);
    }

    @Override
    public TaskWithBlock takeover(final String taskId, final String owner) {
        return doTakeover(taskId, owner);
    }

    @Override
    public Stream<TaskWithBlock> query(final LocalDate begin, final LocalDate end, Paging paging) {
        Paging limited = paginationService.getLimitedPaging(paging);
        ArchivedTaskSearchParameters searchParameters = toSearchParameters(begin, end, true);
        searchParameters.showPage(limited.page(), limited.resultsPerPage());
        paginationService.addPagingHeaderIfNeeded(paging, limited, paginationService.toSetHeader(response),
                () -> taskArchive.countTotalResults(searchParameters));
        return search(searchParameters, false).map(castToTaskWithBlock);
    }

    @Override
    public Stream<TaskWithBlock> export(final LocalDate begin, final LocalDate end, Paging paging) {
        Paging limited = paginationService.getLimitedPaging(paging);
        ArchivedTaskSearchParameters searchParameters = toSearchParameters(begin, end, true);
        searchParameters.showPage(limited.page(), limited.resultsPerPage());
        paginationService.addPagingHeaderIfNeeded(paging, limited, paginationService.toSetHeader(response),
                () -> taskArchive.countTotalResults(searchParameters));
        return search(searchParameters, true).map(castToTaskWithBlock);
    }

    protected Task skip(String taskId, List<String> stepPaths, boolean skip) {
        SerializableTask task = getSerializableTask(taskId);
        final List<BlockPath> paths = stepPaths.stream().map(BlockPath::apply).collect(Collectors.toList());
        checkArgument(task != null, "Could not find active task %s", taskId);
        checkPermission(TASK_SKIPSTEP, task);

        if (skip) {
            engine.get().skipStepPaths(taskId, paths);
        } else {
            engine.get().unskipStepPaths(taskId, paths);
        }
        return engine.get().retrieve(taskId);
    }

    protected void pause(final String taskId, final String stepPath) {
        checkForMissingPermission(isNotCalledByOwner(taskId) && !hasPermission(ADMIN), "Only owner or ADMIN can add pause to the task.");
        engine.get().addPauseStep(taskId, BlockPath.apply(stepPath));
    }

    @Override
    public StepState getStep(String taskId, String stepPath, DateTime ifModifiedSince) {
        TaskWithBlock taskWithBlock = (TaskWithBlock) viewAbleTask(pendingOrLiveOrArchivedTask(taskId, true));
        StepState step = ((Block) taskWithBlock.getBlock()).getStep(BlockPath.apply(stepPath).tail());
        step = addLogs(taskId, BlockPath.apply(stepPath), step);

        if (hasBeenModifiedSince(step, ifModifiedSince)) {
            return step;
        } else {
            throw new NotModifiedException();
        }
    }

    public void setPaginationService(PaginationService paginationService) {
        this.paginationService = paginationService;
    }

    public void setResponse(HttpResponse response) {
        this.response = response;
    }

    @Override
    public void setRoleService(RoleService roleService) {
        this.roleService = roleService;
    }
}
