package com.xebialabs.deployit.taskexecution.security;

import com.google.common.collect.Lists;
import com.xebialabs.xlplatform.distribution.RSAUtils;
import org.apache.http.Header;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.RequestLine;
import org.apache.http.message.BasicHeader;
import org.apache.http.protocol.HttpContext;
import org.bouncycastle.crypto.params.RSAKeyParameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import java.util.UUID;

public class TaskWorkerHttpRequestInterceptor implements HttpRequestInterceptor {
    private static final Logger logger = LoggerFactory.getLogger(TaskWorkerHttpRequestInterceptor.class);

    private static final String RSA_SIGNATURE = "rsa";
    private static final String DELIMITER = ":";

    private final RSAKeyParameters privateKey;
    private final String workerId;

    public TaskWorkerHttpRequestInterceptor(String workerId, RSAKeyParameters privateKey) {
        this.privateKey = privateKey;
        this.workerId = workerId;
    }

    @Override
    public void process(final HttpRequest request, final HttpContext context) {
        final Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication != null) {
            try {
                request.addHeader(createRSASignatureHeader(authentication.getName(), request));
            } catch (Exception e) {
                logger.error("Unable to add RSA signature {}", e.getMessage());
            }
        }
    }

    private Header createRSASignatureHeader(final String name, final HttpRequest request) throws Exception {
        final RequestLine requestLine = request.getRequestLine();
        final String nonce = UUID.randomUUID().toString().concat(DELIMITER).concat(String.valueOf(System.currentTimeMillis()));
        final String message = String.join(DELIMITER, Lists.newArrayList(requestLine.getUri(), requestLine.getMethod(), requestLine.getProtocolVersion().toString(), nonce));
        final String signature = RSAUtils.sign(message, privateKey);
        return new BasicHeader(RSA_SIGNATURE, name.concat(DELIMITER).concat(workerId).concat(DELIMITER).concat(nonce).concat(DELIMITER).concat(signature));
    }
}
