package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemProperties;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.CandidateValuesFilter;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.SearchParameters;
import nl.javadude.scannit.Scannit;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.EnumSet;
import java.util.Optional;
import java.util.Set;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static java.lang.String.format;

public class SearchParametersFactory {
    static final long DEFAULT_RESULTS_PER_PAGE = 25L;

    public static SearchParameters createSearchParams(Type type, long page, long resultPerPage) {
        SearchParameters searchParams = new SearchParameters();

        if (type != null) {
            searchParams.setType(type);
        }

        // If not given, are default to 0.
        searchParams.setPage(page);
        // If different from 0 then set it; 0 means default, negative will mean 'all'.
        if (resultPerPage > 0) {
            searchParams.setResultsPerPage(resultPerPage);
        } else if (resultPerPage == 0) {
            searchParams.setResultsPerPage(DEFAULT_RESULTS_PER_PAGE);
        }

        return searchParams;
    }

    public static SearchParameters createSearchParams(Type typeName, long page, long resultPerPage, String parent, String ancestor, String namePattern, String idPattern, Calendar lastModifiedBefore, Calendar lastModifiedAfter, int depth) {
        SearchParameters searchParams = createSearchParams(typeName, page, resultPerPage);
        if (isNotBlank(parent)) {
            searchParams.setParent(parent);
        }
        if (isNotBlank(ancestor)) {
            searchParams.setAncestor(ancestor);
        }
        if (isNotBlank(namePattern)) {
            searchParams.setName(namePattern);
        }
        if (isNotBlank(idPattern)) {
            searchParams.setId(idPattern);
        }
        searchParams.setBefore(lastModifiedBefore);
        searchParams.setAfter(lastModifiedAfter);
        searchParams.setDepth(depth);

        return searchParams;
    }

    public static SearchParameters createSearchParams(Type typeName, long page, long resultPerPage, String parent, String ancestor, String namePattern, Calendar lastModifiedBefore, Calendar lastModifiedAfter, int depth) {
        return createSearchParams(typeName, page, resultPerPage, parent, ancestor, namePattern, null, lastModifiedBefore, lastModifiedAfter, depth);
    }

    public static SearchParameters createSearchParams(Type typeName, long page, long resultPerPage, String parent, String ancestor,
                                                      String namePattern, String idPattern, Calendar lastModifiedBefore,
                                                      Calendar lastModifiedAfter, int depth, ConfigurationItemProperties properties) {
        SearchParameters searchParameters = createSearchParams(typeName, page, resultPerPage, parent, ancestor, namePattern,
                idPattern, lastModifiedBefore, lastModifiedAfter, depth);

        if (properties != null && !properties.properties().isEmpty()) {
            properties.properties().forEach((k, v) -> searchParameters.getProperties().put(k, v));
        }
        return searchParameters;
    }

    public static SearchParameters createSearchParamsForCandidateValues(String propertyName, String namePattern, String idPattern, long page, long resultPerPage, ConfigurationItem ci) {
        final PropertyDescriptor pd = ci.getType().getDescriptor().getPropertyDescriptor(propertyName);
        Checks.checkNotNull(pd, "Property %s does not exist on type %s", propertyName, ci.getType().getName());
        Checks.checkArgument(EnumSet.of(CI, LIST_OF_CI, SET_OF_CI).contains(pd.getKind()), "Property %s of type %s is not of kind CI, LIST_OF_CI or SET_OF_CI", propertyName, ci.getType().getName());

        Set<Method> filterMethods = Scannit.getInstance().getMethodsAnnotatedWith(CandidateValuesFilter.class);
        Optional<Method> filterMethod = filterMethods.stream().filter(m -> m.getAnnotation(CandidateValuesFilter.class).name().equals(pd.getCandidateValuesFilter())).findFirst();

        SearchParameters params;
        if (filterMethod.isPresent()) {
            try {
                com.xebialabs.deployit.plugin.api.services.SearchParameters apiParams = (com.xebialabs.deployit.plugin.api.services.SearchParameters) filterMethod.get().invoke(null, ci, pd);
                params = new SearchParameters(apiParams);
            } catch (IllegalAccessException e) {
                throw new IllegalStateException(format("Could not access filter method [%s]", pd.getCandidateValuesFilter()), e);
            } catch (InvocationTargetException e) {
                throw new IllegalStateException(format("Could not invoke filter method [%s]", pd.getCandidateValuesFilter()), e);
            }
        } else {
            params = new SearchParameters();
            params.setType(pd.getReferencedType());
        }
        params.setName(namePattern);
        params.setId(idPattern);
        params.setPage(page);
        params.setResultsPerPage(resultPerPage);
        return params;
    }

}
