package com.xebialabs.deployit.core.rest.api;

import java.util.*;
import java.util.stream.Collectors;
import jakarta.ws.rs.core.Context;
import org.jboss.resteasy.spi.HttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.api.InternalSecurityProxy;
import com.xebialabs.deployit.core.api.dto.RolePermissions;
import com.xebialabs.deployit.core.rest.api.support.PaginationSupport;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.api.security.RolePrincipals;
import com.xebialabs.deployit.engine.spi.event.RolePermissionsChangedEvent;
import com.xebialabs.deployit.engine.spi.event.RolePrincipalsChangedEvent;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.VIEW_SECURITY;


@Controller
public class SecurityResource extends AbstractSecuredResource implements InternalSecurityProxy {

    @Context
    private HttpResponse response;

    @Autowired
    private RoleService roleService;

    @Autowired
    private PermissionEditor permissionEditor;

    private DtoWriter dtoWriter = new DtoWriter();
    private DtoReader dtoReader = new DtoReader();

    @Override
    public Boolean exists(String roleName) {
        return roleService.roleExists(roleName);
    }

    @Override
    public List<com.xebialabs.deployit.engine.api.security.Role> list(String id, String rolePattern, Paging paging, Ordering order) {
        if (rolePattern == null && (paging == null || paging.resultsPerPage() == -1) && order == null) {
            return dtoWriter.writeRoles(roleService.getRoles(id));
        }
        PaginationSupport.addTotalCountHeader(roleService.countRoles(id, rolePattern), response);
        return dtoWriter.writeRoles(roleService.getRoles(id, rolePattern, paging, order));
    }

    @Override
    public List<RolePermissions> readRolePermissions(String id, String rolePattern, Paging paging, Ordering order, Boolean includeInherited) {
        checkPermissions(EDIT_SECURITY, VIEW_SECURITY);
        Map<Role, Set<Permission>> rolePermissionMultimap;
        if (rolePattern == null && (paging == null || paging.resultsPerPage() == -1) && order == null) {
            rolePermissionMultimap = permissionEditor.readPermissions(id, includeInherited);
        } else {
            rolePermissionMultimap = permissionEditor.readPermissions(id, rolePattern, paging, order, includeInherited);
            PaginationSupport.addTotalCountHeader(roleService.countRoles(id, rolePattern), response);
        }

        return dtoWriter.writeRolePermissions(rolePermissionMultimap);
    }

    @Override
    public void writeRolePermissions(String id, List<RolePermissions> permissions) {
        checkPermission(EDIT_SECURITY);

        List<Role> roles = roleService.getRoles();
        Map<Role, Set<Permission>> multimap = dtoReader.readRolePermissions(permissions, roles);
        permissionEditor.editPermissions(id, multimap);

        fireRolePermissionsChangedEvent(id, permissions);
    }

    private static void fireRolePermissionsChangedEvent(String id, List<RolePermissions> permissions) {
        List<String> collect = permissions.stream().map(p -> p.getRole().getName() + " => " + p.getPermissions()).collect(Collectors.toList());
        EventBusHolder.publish(new RolePermissionsChangedEvent(id, collect));
    }

    @Override
    public List<RolePrincipals> readRolePrincipals(String rolePattern, Paging paging, Ordering order) {
        checkPermissions(EDIT_SECURITY, VIEW_SECURITY);
        List<Role> multimap;
        if (rolePattern == null && (paging == null || paging.resultsPerPage() == -1) && order == null) {
            multimap = roleService.readRoleAssignments();
        } else {
            multimap = roleService.readRoleAssignments(rolePattern, paging, order);
            long total = roleService.countRoles(-1, rolePattern);
            PaginationSupport.addTotalCountHeader(total, response);
        }

        logger.debug("com.xebialabs.deployit.security.RoleService delivered roles: {}", multimap);
        List<RolePrincipals> assignments = dtoWriter.writeRoleAssignments(multimap);
        logger.debug("Going to return the following roles: {}", assignments);
        return assignments;
    }

    @Override
    public void writeRolePrincipals(List<RolePrincipals> rolePrincipals) {
        checkPermission(EDIT_SECURITY);
        logger.debug("Got the following roles: {}", rolePrincipals);

        // merge all same role principals
        Map<String, RolePrincipals> indexed = new HashMap<>();
        for (RolePrincipals rp : rolePrincipals) {
            String roleName = rp.getRole().getName();
            if (roleName == null || roleName.trim().isEmpty()) {
                throw new IllegalArgumentException("Role name cannot be empty");
            } else if (!indexed.containsKey(roleName)) {
                indexed.put(roleName, rp);
            } else {
                indexed.get(roleName).addPrincipals(rp.getPrincipals());
            }
        }

        List<Role> roles = dtoReader.readRoleAssignments(new ArrayList<>(indexed.values()));

        logger.debug("Going to write the following role assignments: {}", roles);
        roleService.writeRoleAssignments(roles);

        fireRolePrincipalsChangedEvent(rolePrincipals);
    }

    private static void fireRolePrincipalsChangedEvent(List<RolePrincipals> rolePrincipals) {
        EventBusHolder.publish(new RolePrincipalsChangedEvent(rolePrincipals.stream().map(rp -> rp.getRole().getName() + " => " + rp.getPrincipals()).collect(Collectors.toList())));
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class UnknownPermissionException extends DeployitException {
        public UnknownPermissionException(String permission) {
            super("Permission %s does not exist.", permission);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(SecurityResource.class);

    public void setRoleService(RoleService roleService) {
        this.roleService = roleService;
    }
}




