package com.xebialabs.deployit.setup

import com.xebialabs.deployit.setup.UpgradeHelper._
import com.xebialabs.xlplatform.io.FolderChecksum.listAllFilesSorted
import grizzled.slf4j.Logging
import org.apache.commons.lang3.StringUtils

import java.io.File
import java.util.Objects
import java.util.regex.Pattern
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

object UpgradeHelper {

  case class DiscoveredPlugin(fileName: String, var path: String)

  val official1000XldPlugins = Seq(
    "glassfish-plugin",
    "iis-plugin",
    "jbossas-plugin",
    "jbossdm-plugin",
    "osb-plugin",
    "tomcat-plugin",
    "wmq-plugin",
    "wps-plugin",
    "was-plugin",
    "xld-aws-plugin",
    "xld-azure-plugin",
    "xld-chef-plugin",
    "xld-cloud-foundry-plugin",
    "xld-cyberark-conjur-plugin",
    "xld-docker-plugin",
    "xld-hashicorp-vault-plugin",
    "xld-kubernetes-plugin",
    "xld-openshift-plugin",
    "xld-puppet-plugin",
    "xld-terraform-plugin",
    "xld-oracle-soa-plugin",
    "vsphere-plugin",
    "windows-plugin",
    "xld-codepipeline-plugin",
    "biztalk-plugin",
    "netscaler-plugin",
    "xld-datapower-plugin",
    "wlp-plugin",
    "wls-plugin",
    "xld-helm-plugin",
    "xld-terraform-enterprise-plugin",
    "bigip-plugin")

  val PluginsFolderName = "plugins"
  val LocalPluginsFolder = "__local__"

  // windows file separator is \ and this needs to be escaped with another \
  val RegexSafeFileSeparator: String = Try(Pattern.compile(File.separator)) match { case Success (_) => File.separator; case Failure(_) => File.separator + File.separator}
}

class UpgradeHelper(previousInstallation: String, nonInteractiveMode: Boolean = false, fileCopier: FileCopier) extends Logging {

  def copyData(files: java.util.List[String], dirs: java.util.List[String],
               filesToMention: java.util.List[String]): Unit = {
    copyData(files, dirs, filesToMention, java.util.List.of())
  }
  /**
    * Copy files, directories passed to a method.
    * It also copied plugins directory skipping plugins that have newer version.
    *
    * @param files          list of mandatory files to copy(replace if exist) them one by one
    * @param dirs           list of directories to copy(replace if exist) as is
    * @param filesToMention list of files to mention that user should adapt and copy them manually
    * @param optionalFiles  list of optional files to copy(replace if exists) them one by one.
    */
  def copyData(files: java.util.List[String], dirs: java.util.List[String],
               filesToMention: java.util.List[String], optionalFiles: java.util.List[String]): Unit = {
    validatePreviousInstallationLocation()
    println(s"\nGoing to copy configuration from previous installation [$previousInstallation].")

    val actions: List[CopyAction] =
      copyFiles(files.asScala.toList, optionalFiles.asScala.toList) ::
        copyDirectories(dirs.asScala.toList) :::
        copyPlugins()

    logger.info(s"Going to copy configuration from previous installation [$previousInstallation].")
    actions.foreach(_.execute())
    logger.info("Copied selected files from previous installation.")

    println("\nCopied selected files from previous installation.")
    mentionFiles(filesToMention.asScala.toList)
  }

  def mentionFiles(files: List[String]): Unit = {
    println("\n\n=> UPGRADE FILES MANUALLY\n")
    println("The following files could not be upgraded automatically and were not copied.")
    println("- Database driver inside the lib directory")
    files.foreach { file =>
      println(s"- $file")
    }
    if (files.nonEmpty) {
      println("")
      println("Please revise the differences between the old and new versions and apply the needed changes manually.\n")
      println("Consult the documentation or contact support for more information.")
    }
  }

  /**
    * Copy files from existing to the new installation.
    * Ask user for a confirmation if setup is in interactive mode.
    *
    * @param files          list of mandatory file names relative to the installation folder
    * @param optionalFiles  list of optional file names relative to the installation folder
    */
  def copyFiles(files: List[String], optionalFiles: List[String]): CopyAction = {
    val allFiles = List.concat(files, optionalFiles)
    val fileNames = allFiles.mkString(", ")
    printIfInteractive(s"Do you want to copy files to the new installation?")
    printIfInteractive(s"Files to be copied [$fileNames]")
    askAndCopy(fileNames, "files") {
      allFiles.foreach(file => fileCopier.tryCopy(previousInstallation, file, optionalFiles.contains(file)))
    }
  }

  /**
    * Copy directories from existing to the new installation.
    * Ask user for a confirmation if setup is in interactive mode.
    *
    * @param dirs list of directory names relative to the installation folder
    */
  def copyDirectories(dirs: List[String]): List[CopyAction] = {
    dirs.map { directory =>
      val previousDirectory = new File(previousInstallation, directory)
      printIfInteractive(s"Do you want to copy directory [$previousDirectory] to the new installation?")
      askAndCopy(directory, "directory") {
        fileCopier.tryCopy(previousInstallation, directory)
      }
    }
  }

  type UpdatedPlugin = (String, String, String)

  private def validatePreviousInstallationLocation(): Unit = {
    val previousInstallationDir = new File(previousInstallation)
    if (!previousInstallationDir.exists || !previousInstallationDir.isDirectory) {
      throw new RuntimeException(s"Provided previous installation location [$previousInstallation] does not exist or is not a directory.")
    }
  }

  private def printIfInteractive(message: String): Unit = if (!nonInteractiveMode) println(message)

  private val supportedPluginExtensions = List(".jar", ".xldp", ".zip")

  /* SemVer regexp taken from https://github.com/semver/semver/blob/master/semver.md  */
  private val semver = """^(.*?)((0|[1-9]\d*)\.(0|[1-9]\d*)\.(0|[1-9]\d*)(?:-((?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*)(?:\.(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*))*))?(?:\+([0-9a-zA-Z-]+(?:\.[0-9a-zA-Z-]+)*))?)$""".r

  private[setup] def findMissingPlugins(oldPluginNames: List[DiscoveredPlugin],
                                        newPluginNames: List[DiscoveredPlugin],
                                        updatedPlugins: List[UpdatedPlugin]
                                       ): List[DiscoveredPlugin] = {
    def findNotUpdated(names: List[DiscoveredPlugin]): List[DiscoveredPlugin] =
      names.filterNot(p =>
        updatedPlugins.map(_._1).exists(s => p.fileName.startsWith(s))
      )

    val newPluginsNotUpdated = findNotUpdated(newPluginNames)
    findNotUpdated(oldPluginNames).filterNot(oldPlugin =>
      newPluginsNotUpdated.exists(newPlugin => newPlugin.fileName.equals(oldPlugin.fileName)))
  }

  def extractVersion(pluginName: String): Option[(String, String)] = pluginName match {
    case semver(commonName, version, _*) => Some((commonName, version))
    case _ => None
  }

  private[setup] def findUpdatedPlugins(oldPlugins: List[DiscoveredPlugin], currentPlugins: List[DiscoveredPlugin]): List[UpdatedPlugin] = {
    val versions = new ListBuffer[UpdatedPlugin]()

    def addIfUpdated(oldPlugin: Option[(String, String)], newPlugin: Option[(String, String)]): Unit = {
      (oldPlugin, newPlugin) match {
        case (Some((oldName, oldVersion)), Some((newName, newVersion))) =>
          if (Objects.equals(oldName, newName) && !Objects.equals(oldVersion, newVersion)) {
            versions += ((StringUtils.chop(oldName), oldVersion, newVersion))
          }
        case _ =>
      }
    }

    def comparePlugins(oldPlugin: DiscoveredPlugin, newPlugin: DiscoveredPlugin): Unit = {
      supportedPluginExtensions.foreach { extension =>
        if (oldPlugin.fileName.endsWith(extension) && newPlugin.fileName.endsWith(extension)) {
          val oldExtracted = extractVersion(oldPlugin.fileName.replace(extension, ""))
          val newExtracted = extractVersion(newPlugin.fileName.replace(extension, ""))
          addIfUpdated(oldExtracted, newExtracted)
        }
      }
    }

    for (old <- oldPlugins) {
      for (current <- currentPlugins) {
        comparePlugins(old, current)
      }
    }
    versions.toList
  }

  private def copyPlugins(): List[CopyAction] = {
    val previousPlugins = new File(previousInstallation, PluginsFolderName)
    logger.debug(s"Going to copy [$previousPlugins] from previous installation.")

    def listPlugins(file: File): List[DiscoveredPlugin] = listAllFilesSorted(file)
      .filter(p => supportedPluginExtensions.exists(e => p.getName.endsWith(e)))
      .map(plugin => DiscoveredPlugin(
        plugin.getName,
        file.toPath.relativize(plugin.toPath).toFile.getPath))

    val oldPlugins = listPlugins(previousPlugins)
    val newPlugins = listPlugins(new File(PluginsFolderName))
    val updatedPlugins = findUpdatedPlugins(oldPlugins, newPlugins)

    val missingPlugins: List[DiscoveredPlugin] = findMissingPlugins(oldPlugins, newPlugins, updatedPlugins)
    missingPlugins.map { plugin =>
      handleMissingPlugin(plugin, newPlugins.head.path.split(RegexSafeFileSeparator)(0))
    } :::
      updatedPlugins.map { case (name, oldVersion, newVersion) =>
        new LogNoOp(s"Plugin [$name] has been updated [$oldVersion -> $newVersion]. Not copying it.")
      }
  }

  def stripExtension(filename: String): String = filename.substring(0, filename.lastIndexOf('.'))

  private[setup] def handleMissingPlugin(plugin: DiscoveredPlugin, officialPluginsSubfolderName: String): CopyAction = {
    def resolveDestinationSubfolderName(pluginFileName: String): String = {
      val LocalPluginsFolder = "__local__"
      extractVersion(pluginFileName) match {
        case Some((name, _)) => if (official1000XldPlugins.contains(StringUtils.chop(name))) officialPluginsSubfolderName else LocalPluginsFolder
        case None => LocalPluginsFolder
      }
    }

    // if plugin is located in a "plugins" folder, and needs to be moved into plugins/xlr-official, plugins/xld-official or plugins/__local__.
    // This is for all clients upgrading from any pre 10.1.0 version to 10.1.0 or later.
    val isPluginFileInRootPluginsDirectory = !plugin.path.contains(File.separator)
    val destinationSubfolderName = if (isPluginFileInRootPluginsDirectory) resolveDestinationSubfolderName(stripExtension(plugin.fileName)) else plugin.path.split(RegexSafeFileSeparator)(0)

    val pluginName = s"$PluginsFolderName${File.separator}${plugin.path}"
    printIfInteractive(s"Plugin [$pluginName] is missing in the new installation. Do you want to copy it?")

    new PluginAskAndCopyStrategy(nonInteractiveMode, plugin, pluginName, destinationSubfolderName, previousInstallation, fileCopier).createCopyAction()
  }

  private def askAndCopy(name: String, fileType: String)(action: => Unit): CopyAction = {
    new FileAskAndCopyStrategy(nonInteractiveMode, fileType, name, action).createCopyAction()
  }

}
