package com.xebialabs.deployit.jetty;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.SessionCookieConfig;

/**
 * {@link ServletContextListener} that can be used to configure the session cookie sent by the web server.
 * It takes care of setting the path of the cookie to the context path.
 */
public class SessionCookieSetupListener implements ServletContextListener {
    private static final Logger LOG = LoggerFactory.getLogger(SessionCookieSetupListener.class);

    private final String sessionCookieName;
    private final String cookieDomain;
    private final boolean secureCookie;
    private final boolean httpOnly;
    private int maxAgeInSeconds;

    public SessionCookieSetupListener(String sessionCookieName, String cookieDomain, boolean secureCookie, boolean httpOnly, int maxAgeInSeconds) {
        this.sessionCookieName = sessionCookieName;
        this.cookieDomain = cookieDomain;
        this.secureCookie = secureCookie;
        this.httpOnly = httpOnly;
        this.maxAgeInSeconds = maxAgeInSeconds;
    }

    public SessionCookieSetupListener(String sessionCookieName, boolean secureCookie, boolean httpOnly, int maxAgeInSeconds) {
        this.sessionCookieName = sessionCookieName;
        this.cookieDomain = null;
        this.secureCookie = secureCookie;
        this.httpOnly = httpOnly;
        this.maxAgeInSeconds = maxAgeInSeconds;
    }

    @Override
    public void contextInitialized(ServletContextEvent sce) {
        ServletContext servletContext = sce.getServletContext();
        SessionCookieConfig scf = servletContext.getSessionCookieConfig();

        String cookiePath = determineCookiePath(servletContext);

        scf.setName(sessionCookieName);
        scf.setPath(cookiePath);

        if (cookieDomain != null) {
            scf.setDomain(cookieDomain);
        }
        scf.setSecure(secureCookie);
        scf.setHttpOnly(httpOnly);
        scf.setMaxAge(maxAgeInSeconds);

        LOG.info("Configuring session cookie: name={} path={} domain={} secure={} httpOnly={} maxAge={}", sessionCookieName, cookiePath, cookieDomain, secureCookie, httpOnly, maxAgeInSeconds);
    }

    protected String determineCookiePath(ServletContext servletContext) {
        String cookiePath = servletContext.getContextPath();
        if ("".equals(cookiePath)) {
            return "/";
        } else {
            return cookiePath;
        }
    }

    @Override
    public void contextDestroyed(ServletContextEvent sce) {
    }
}
