package com.xebialabs.ascode.yaml.sugar

import com.xebialabs.ascode.utils.TypeSugar.typeOf
import com.xebialabs.ascode.yaml.sugar.GenerateStrategy.Generate
import com.xebialabs.deployit.plugin.api.reflect.Type

import scala.reflect.ClassTag

sealed trait Action

case class Add(fieldName: String, value: String) extends Action

case class Change(originalFieldName: String, replaceFieldName: String) extends Action

case class Remove(fieldName: String) extends Action

sealed trait GenerateStrategy

object GenerateStrategy {
  case object Reject extends GenerateStrategy

  case class Filter(condition: Type => Boolean) extends GenerateStrategy

  object Skip extends Filter(_ => false)

  object Generate extends Filter(_ => true)
}

case class GenerateHints(idField: Option[String] = None, strategy: GenerateStrategy = Generate)

case class SugarDescriptor(ciType: Type,
                           key: Option[String],
                           deSugarActions: List[Action],
                           sugarActions: List[Action],
                           generateHints: GenerateHints,
                           whitelistedFields: Set[String] = Set.empty) {
  def findSugaringAction(name: String): Option[Action] = sugarActions.find {
    case Add(fieldName, _) => fieldName == name
    case Change(originalFieldName, _) => originalFieldName == name
    case Remove(fieldName) => fieldName == name
  }
}

object SugarConfig {
  private def reverseAction = (action: Action) => action match {
    case Add(fieldName, _) => Some(Remove(fieldName))
    case _: Remove => None
    case Change(originalFieldName, replaceFieldName) => Some(Change(replaceFieldName, originalFieldName))
  }

  def descriptor[T](key: String)(implicit classTag: ClassTag[T]): SugarDescriptor = {
    descriptor[T](Some(key))
  }

  def descriptorFromType(ciType: Type,
                         key: Option[String] = None,
                         deSugarActions: List[Action] = Nil,
                         sugarActions: List[Action] = Nil,
                         generateIdField: Option[String] = None,
                         generateStrategy: GenerateStrategy = Generate,
                         blacklistedFields: Set[String] = Set(),
                         whitelistedFields: Set[String] = Set()): SugarDescriptor = {

    val shouldGenerateActions = key.isDefined && sugarActions.isEmpty && deSugarActions.isEmpty
    val finalDeSugarActions = if (shouldGenerateActions) List(Add("type", ciType.toString), Change(key.get, "name")) else deSugarActions
    val finalSugarActions = if (shouldGenerateActions) finalDeSugarActions.flatMap(reverseAction(_)) else sugarActions
    SugarDescriptor(ciType,
      key,
      finalDeSugarActions,
      blacklistedFields.map(Remove).toList ::: finalSugarActions,
      GenerateHints(generateIdField, generateStrategy),
      whitelistedFields
    )
  }

  def descriptor[T](key: Option[String] = None,
                    deSugarActions: List[Action] = Nil,
                    sugarActions: List[Action] = Nil,
                    generateIdField: Option[String] = None,
                    generateStrategy: GenerateStrategy = Generate,
                    blacklistedFields: Set[String] = Set(),
                    whitelistedFields: Set[String] = Set())(implicit classTag: ClassTag[T]): SugarDescriptor = {
    val ciType = typeOf[T]

    descriptorFromType(ciType, key, deSugarActions, sugarActions, generateIdField, generateStrategy, blacklistedFields, whitelistedFields)
  }

  def apply(descriptors: SugarDescriptor*): SugarConfig = {
    val init = (Map[String, SugarDescriptor](), Map[Type, SugarDescriptor]())
    val (keyToDescriptor, typeToDescriptor) = descriptors.foldLeft(init) { case ((keyAcc, typeAcc), current) =>
      (current.key.map(key => keyAcc + (key -> current)).getOrElse(keyAcc), typeAcc + (current.ciType -> current))
    }
    SugarConfig(typeToDescriptor, keyToDescriptor)
  }
}

case class SugarConfig(typeToDescriptors: Map[Type, SugarDescriptor],
                       keysToDescriptors: Map[String, SugarDescriptor])