package com.xebialabs.ascode.yaml.writer

import java.io.OutputStream
import java.util.Properties
import java.util.zip.{ZipEntry, ZipOutputStream}

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.Specs
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.deployit.core.rest.YamlSupport
import com.xebialabs.deployit.io.ArtifactFile
import org.apache.commons.io.IOUtils

import scala.jdk.CollectionConverters._

object DefinitionWriter {
  def apply(product: String, specGenerators: (List[String], SpecWriter)*): DefinitionWriter = {
    new DefinitionWriter(Specs(specGenerators.flatMap { case (kinds, parser) =>
      kinds.map(kind => (product -> kind) -> parser)
    }.toMap))
  }

  case class WriterConfig(definitions: List[Definition], writeSecrets: Boolean = false, writeDefaults: Boolean = false)
}

class DefinitionWriter(specs: Specs[SpecWriter]) extends YamlSupport {
  private def getSpecWriter(definition: Definition) = {
    val project = definition.apiVersion.split('/').head
    specs.getSpec(project, definition.kind)
  }

  private def writeYaml(stream: ZipOutputStream, definitions: List[Definition])(implicit config: WriterConfig) = {
    stream.putNextEntry(new ZipEntry("index.yaml"))

    val context = definitions.map { definition =>
      val yamlGenerator = yamlFactory.createGenerator(stream)

      if (definition.apiVersion.isEmpty || definition.apiVersion.indexOf('/') == -1) {
        throw new AsCodeException(s"apiVersion format invalid: ${Option(definition.apiVersion).getOrElse("")}")
      }

      yamlGenerator.writeStartObject()
      yamlGenerator.writeStringField("apiVersion", definition.apiVersion)
      yamlGenerator.writeStringField("kind", definition.kind)

      yamlGenerator.writeFieldName("spec")
      val generateContext = getSpecWriter(definition).write(definition.spec, yamlGenerator)
      yamlGenerator.writeEndObject()
      yamlGenerator.flush()
      stream.write("\n".getBytes)
      generateContext
    }
    stream.closeEntry()
    context
  }

  private def writeSecrets(contexts: List[GenerateContext])
                            (implicit config: WriterConfig, zipStream: ZipOutputStream): Unit = {
    if (config.writeSecrets) {
      val allSecrets = contexts.flatMap(_.secrets).toMap
      if (allSecrets.nonEmpty) {
        zipStream.putNextEntry(new ZipEntry("secrets.xlvals"))
        val properties = new Properties
        properties.putAll(allSecrets.asJava)
        properties.store(zipStream, "Secrets")
        zipStream.closeEntry()
      }
    }
  }

  private def writeAdditionalFiles(contexts: List[GenerateContext])(implicit zipStream: ZipOutputStream): Unit = {
    contexts.flatMap(_.additionalFile).foreach(additionalFile => {
      zipStream.putNextEntry(new ZipEntry(additionalFile.name))
      val stream = additionalFile.file match {
        case file: ArtifactFile => file.getRawStream
        case _@file => file.getInputStream
      }
      IOUtils.copy(stream, zipStream)
      stream.close()
      zipStream.closeEntry()
    })
  }

  private def writeZipFile(stream: OutputStream)(implicit config: WriterConfig): Unit = {
    val definitions = config.definitions
    definitions.foreach(definition => getSpecWriter(definition).validate(definition.spec))
    implicit val zipStream: ZipOutputStream = new ZipOutputStream(stream)
    val contexts = writeYaml(zipStream, definitions)
    writeAdditionalFiles(contexts)
    writeSecrets(contexts)
    zipStream.close()
  }

  def write(stream: OutputStream, config: WriterConfig): Unit = writeZipFile(stream)(config)
}
