package com.xebialabs.deployit.ascode.service.spec

import com.xebialabs.ascode.service.spec.{InterpreterContext, SpecInterpreter}
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.ascode.yaml.model.UsersSpec
import com.xebialabs.ascode.yaml.model.{User => XLDUser}
import com.xebialabs.deployit.core.rest.api.UserResource
import com.xebialabs.deployit.engine.api.security.User
import com.xebialabs.deployit.exception.NotFoundException
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@Autowired
class UsersSpecInterpreter(userResource: UserResource) extends SpecInterpreter with Logging {
  override def isDefinedAt(context: InterpreterContext): Boolean = context.definition.spec.isInstanceOf[UsersSpec]

  override def apply(context: InterpreterContext): AsCodeResponse = {
    val spec = context.definition.spec.asInstanceOf[UsersSpec]
    AsCodeResponse.ids(spec.users.foldLeft(USER.ids) {
      case (acc, user: XLDUser) =>
        try {
          val foundUser = userResource.read(user.username)
          foundUser.setPassword(user.password)
          userResource.modifyPassword(user.username, foundUser)
          acc.withUpdated(user.username)
        } catch {
          case _: NotFoundException =>
            logger.warn(s"User not found so creating user ${user.username}")
            userResource.create(user.username, UserUtils.asCodeUserToUser(user))
            acc.withCreated(user.username)
        }
      case (acc, _) => acc
    })
  }
}

object UserUtils {
  def asCodeUserToUser(xldUser: XLDUser): User = {
    val user = new User(xldUser.username, false)
    user.setPassword(xldUser.password)
    user
  }
}
