package com.xebialabs.deployit.ascode.service

import com.xebialabs.ascode.yaml.dto.PreviewResponse.TaskPreviewItem
import com.xebialabs.deployit.core.rest.view.{InvocationDescriptorView, PreviewResult}
import com.xebialabs.deployit.engine.api.dto.Deployment
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.collection.mutable.ListBuffer

@Service
@Autowired
class StitchPreviewService {
  def transformStitchPreviewResult(previewResult: PreviewResult): List[TaskPreviewItem] = {
    val taskPreviewItemList = ListBuffer.empty[TaskPreviewItem]
    if (previewResult.getDeployedMap.isEmpty || !containsTransformations(previewResult)) {
      taskPreviewItemList += TaskPreviewItem("There are no stitch transformations for given deployment", Nil)
    } else {
      previewResult.getDeployedMap.forEach((deployedName, invocations) => {
        taskPreviewItemList += TaskPreviewItem(s"Deployed [$deployedName]", transformDeployedInvocations(invocations.asScala.toList))
      })
    }
    taskPreviewItemList.toList
  }

  private def transformDeployedInvocations(invocations: List[InvocationDescriptorView]): List[TaskPreviewItem] = {
    invocations.zipWithIndex.map(invocation => {
      TaskPreviewItem(s"Invocation ${invocation._2 + 1}", transformInvocationView(invocation._1))
    }).sortBy(_.name)
  }

  private def transformInvocationView(invocation: InvocationDescriptorView): List[TaskPreviewItem] = {
    val taskPreviewItemList = ListBuffer.empty[TaskPreviewItem]
    val processorList = ListBuffer.empty[TaskPreviewItem]
    val diffLineList = ListBuffer.empty[TaskPreviewItem]
    invocation.getDiff.split("\n").foreach(line => {
      diffLineList += TaskPreviewItem(line, Nil)
    })
    invocation.getProcessors.asScala.toList.zipWithIndex.foreach(processor => {
      processorList += TaskPreviewItem(s"${processor._2 + 1}. ${processor._1.getDescription}", getProcessorErrors(invocation.getErrors.asScala.getOrElse(processor._1.getId.toString, "")))
    })
    taskPreviewItemList += TaskPreviewItem("Diff between input and output", diffLineList.toList)
    taskPreviewItemList += TaskPreviewItem("Processors", processorList.toList)
    taskPreviewItemList.toList
  }

  private def getProcessorErrors(error: String): List[TaskPreviewItem] = {
    val processorError = ListBuffer.empty[TaskPreviewItem]
    if (error.nonEmpty) {
      error.split("\n").foreach(line => {
        processorError += TaskPreviewItem(line, Nil)
      })
    }
    processorError.toList
  }

  private def containsTransformations(previewResult: PreviewResult): Boolean = {
    var containsTransformations = false
    previewResult.getDeployedMap.forEach((_, invocations) => {
      invocations.forEach(invocation => {
        if (!invocation.getProcessors.isEmpty) {
          containsTransformations = true
        }
      })
    })
    containsTransformations
  }

  def getStitchTaskDescription(deployment: Deployment,msg : String): String = {
      deployment.getDeployedApplication match {
        case deployedApp: DeployedApplication =>
          s"$msg '${deployedApp.getVersion.getId}' to '${deployedApp.getEnvironment.getId}'"
      }
    }
}
