package com.xebialabs.deployit.core.rest

import java.io._

import com.xebialabs.ascode.annotation.ExceptionToAsCodeResponse
import com.xebialabs.ascode.service.spec.InterpreterContext
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.deployit.ascode.repository.WorkDirConstants._
import com.xebialabs.deployit.ascode.schema.JsonSchemaGenerator
import com.xebialabs.deployit.ascode.service.generator.DefinitionGeneratorService
import com.xebialabs.deployit.ascode.service.generator.DefinitionGeneratorService.GeneratorConfig
import com.xebialabs.deployit.ascode.service.{ArchiveProcessor, DefinitionInterpreterService, PreviewService}
import com.xebialabs.deployit.ascode.yaml.sugar.XLDSugar.defaultConfig
import com.xebialabs.deployit.core.rest.resteasy.Workdir
import com.xebialabs.deployit.core.rest.resteasy.Workdir.Clean.ONLY_ON_EXCEPTION
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import grizzled.slf4j.Logging
import jakarta.ws.rs.core.Response
import jakarta.ws.rs.{Consumes, POST, Path, _}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

@Controller
@ExceptionToAsCodeResponse
@Path("/devops-as-code")
class XLDAsCodeResource @Autowired()(asCodeInterpretationService: DefinitionInterpreterService,
                                     previewService: PreviewService,
                                     archiveProcessor: ArchiveProcessor,
                                     generatorService: DefinitionGeneratorService,
                                     jsonGenerator: JsonSchemaGenerator) extends Logging {
  @POST
  @Path("/apply")
  @Workdir(prefix = AS_CODE_APPLY_PREFIX, clean = ONLY_ON_EXCEPTION)
  @Produces(Array("application/json"))
  def interpret(definition: Definition, scmData: Option[SCMTraceabilityData]): Response = {
    val response = asCodeInterpretationService.interpret(InterpreterContext(definition, scmData))
    Response.ok(response).build()
  }

  @POST
  @Path("/apply")
  @Consumes(Array("application/zip"))
  @Produces(Array("application/json"))
  @Workdir(prefix = AS_CODE_APPLY_PREFIX, clean = ONLY_ON_EXCEPTION)
  def interpretZip(in: InputStream, scmData: Option[SCMTraceabilityData]): Response = {
    val response = archiveProcessor.process(in, scmData)
    Response.ok(response).build()
  }

  @POST
  @Path("/preview")
  @Workdir(prefix = AS_CODE_APPLY_PREFIX, clean = ONLY_ON_EXCEPTION)
  @Produces(Array("application/json"))
  def preview(definition: Definition): Response = {
    val response = previewService.preview(definition)
    Response.ok(response).build()
  }

  @GET
  @Path("/generate")
  @Produces(Array("application/zip"))
  @Workdir(prefix = AS_CODE_GENERATE_PREFIX, clean = ONLY_ON_EXCEPTION)
  def generate(@QueryParam("path") anchestor: String,
               @DefaultValue("false") @QueryParam("globalPermissions") globalPermissions: Boolean,
               @DefaultValue("false") @QueryParam("roles") roles: Boolean,
               @DefaultValue("false") @QueryParam("users") users: Boolean,
               @DefaultValue("false") @QueryParam("secrets") includeSecrets: Boolean,
               @DefaultValue("false") @QueryParam("defaults") includeDefaults: Boolean): Response = {
    val config = GeneratorConfig(Option(anchestor).filter(_.nonEmpty), globalPermissions, roles, users, includeSecrets)
    val definitions = generatorService.generate(config)
    Response.ok(WriterConfig(definitions, includeSecrets, includeDefaults)).build()
  }

  @GET
  @Path("/schema")
  @Produces(Array("application/json"))
  def schema(): Response = {
    Response.ok(jsonGenerator.YamlSchema).build()
  }
}
