package com.xebialabs.deployit.ascode.schema

import com.xebialabs.ascode.schema.JsonSchema
import com.xebialabs.ascode.schema.JsonSchema.{SchemaNode, _}
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.deployit.ascode.yaml.model.Constants
import com.xebialabs.deployit.ascode.yaml.model.Constants.Kinds._
import com.xebialabs.deployit.ascode.yaml.model.Constants.Versions._
import com.xebialabs.deployit.ascode.yaml.parser.XLDDefinitionParser
import com.xebialabs.deployit.ascode.yaml.sugar.XLDSugar.defaultConfig
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.reflect._
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.repository.{RepositoryService, SearchParameters}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.collection.immutable.ListMap
import scala.jdk.CollectionConverters._

object JsonSchemaGenerator {
  val prefix: String = buildPrefix(Constants.Schema.PRODUCT_PREFIX)

  private def checkAndGetFileProperty(t: Type) =
    if (t.instanceOf(Type.valueOf(classOf[Artifact]))) {
      ListMap(
        "file" -> node
          .`type`("string")
          .description("Path to a file or folder")
      )
    } else {
      ListMap()
    }

  private def getAllRequiredProperties(pd: List[PropertyDescriptor]) = pd.filter(pd => pd.isRequired &&
    !pd.isAsContainment &&
    (pd.getDefaultValue == null || pd.getDefaultValue.equals("")))
    .map(_.getName)

  private def getStandardProperties(t: Type) = ListMap(
    "name" -> node.`type`("string").description("The name of the CI"),
    "type" -> node.`type`("string").description("The type of the CI").enum(t.toString)
  )

  private def checkAndGetChildrenProperty(t: Type)(implicit generatorContext: SchemaGeneratorContext) =
    if (generatorContext.typeContainmentMap.contains(t)) {
      val types = generatorContext.typeContainmentMap(t).toList
      val childTypes = buildRefList(types, Constants.Schema.PRODUCT_PREFIX)
      ListMap(
        "children" -> node
          .`type`("array")
          .description("The children")
          .items(node.oneOf(childTypes))
      )
    } else {
      ListMap()
    }

  private def kindToJsonType(pd: PropertyDescriptor) = pd.getKind match {
    case BOOLEAN => node.`type`("boolean")
    case INTEGER => node.`type`("integer")
    case ENUM => node.`type`("string").enum(pd.getEnumValues.asScala.toSeq: _*)
    case xs@(LIST_OF_STRING | SET_OF_STRING) => node
      .`type`("array")
      .uniqueItems(xs != LIST_OF_STRING)
      .items(node.`type`("string"))
    case MAP_STRING_STRING => node
      .`type`("object")
      .patternProperties(".*" -> node.`type`("string"))
    case xs@(LIST_OF_CI | SET_OF_CI) =>
      val collectionNode = node.`type`("array")
      val referencedType = pd.getReferencedType
      val subtypes = DescriptorRegistry.getSubtypes(referencedType).asScala.toList
      if (pd.isAsContainment) {
        collectionNode.items(node.oneOf(buildRefList(referencedType :: subtypes, Constants.Schema.PRODUCT_PREFIX)))
      } else {
        collectionNode
          .items(node.`type`("string"))
          .uniqueItems(xs != LIST_OF_CI)
      }
    case _ => node.`type`("string")
  }

  private def getPropertyDefinition(prop: PropertyDescriptor) = node
    .description(prop.getDescription)
    .mergePart(kindToJsonType(prop))
    .mergePart(
      Option(prop.getDefaultValue)
        .map(v => ListMap("default" -> v))
        .getOrElse(ListMap.empty)
    )

  private def getPropertyDefinitions(pd: List[PropertyDescriptor]) =
    ListMap(pd.map(p => (p.getName, getPropertyDefinition(p))): _*)

  def getTypeDefinition(d: Descriptor)(implicit generatorContext: SchemaGeneratorContext): SchemaNode = {
    val pds = d.getPropertyDescriptors.asScala.filterNot(_.isHidden).toList
    val fileProperty = checkAndGetFileProperty(d.getType)

    node
      .`type`("object")
      .description(Option(d.getDescription).getOrElse("No description available"))
      .additionalProperties(false)
      .required("type" :: "name" :: fileProperty.keys.toList ::: getAllRequiredProperties(pds): _*)
      .properties(
        getStandardProperties(d.getType) ++
          getPropertyDefinitions(pds) ++
          checkAndGetChildrenProperty(d.getType) ++
          fileProperty
      )
  }
}

@Service
class JsonSchemaGenerator @Autowired()(repository: RepositoryService,
                                       parser: XLDDefinitionParser) {

  import JsonSchemaGenerator._

  private implicit val generatorContext: SchemaGeneratorContext =
    SchemaGeneratorContext(DescriptorRegistry.getDescriptors.asScala.toList)

  private val allRoots: List[String] = {
    val params = new SearchParameters()
    params.setType(typeOf[Root])
    repository.list(params).asScala.map(_.getId).toList
  }

  private val generators =
    parser
      .specs
      .specs
      .flatMap {
        case ((_, kind), generator: SchemaGenerator) => Some(kind -> generator)
        case _ => None
      }

  private def kindNodes() =
    generators
      .map { case (kind, generator) => generator.generateSchema(kind) }
      .toList

  private def generateDefinitions() =
    generators
      .flatMap { case (_, generator) => generator.generateDefinitions }

  private val xldSchema: SchemaNode = node
    .id("https://xebialabs.com/ascode.schema.json")
    .`type`("object")
    .schema("https://json-schema.org/draft-07/schema#")
    .description("A description of objects in the XL Products")
    .oneOf(importNode() :: kindNodes())
    .definitions(getAllDefinitions ++ generateDefinitions())
  val YamlSchema: SchemaNode = JsonSchema.buildSchemaMerge("xld", xldSchema)

  private def importNode() = node
    .additionalProperties(false)
    .required("apiVersion", "kind", "metadata")
    .properties(
      version(API_VERSION),
      kind(IMPORT),
      metadata(
        node
          .additionalProperties(false)
          .required("imports")
          .properties(imports())
      )
    )

  private def getAllDefinitions =
    allRoots.flatMap(getDirectoryDefinition).toMap ++
      ListMap(generatorContext.allTypes.map(desc => (s"$xlNamespace.${desc.getType.toString}", getTypeDefinition(desc))): _*)

  private def getDirectoryDefinition(root: String) = {
    val dirTypes = buildRefListForRoot(root, Constants.Schema.PRODUCT_PREFIX)

    val dirRef = refNode(s"$prefix.core.Directory.$root")
    val dirRefSugar = refNode(s"$prefix.core.Directory.$root.sugar")
    val allDirRefs = dirRef :: dirRefSugar :: dirTypes

    val childrenNode = node
      .`type`("array")
      .description("The children")
      .items(node.oneOf(allDirRefs))

    definitions(
      s"$xlNamespace.core.Directory.$root" -> node
        .`type`("object")
        .description("Directory")
        .additionalProperties(false)
        .required("type", "name")
        .properties(getStandardProperties(Type.valueOf(classOf[Directory])) + ("children" -> childrenNode)),

      s"$xlNamespace.core.Directory.$root.sugar" -> node
        .`type`("object")
        .description("Directory")
        .additionalProperties(false)
        .required("directory")
        .properties(
          "directory" -> node
            .`type`("string")
            .description("Name of the directory"),
          "children" -> childrenNode
        )
    )
  }
}

