package com.xebialabs.deployit.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.deployit.ascode.service.TempFileMapper.ZipFileEntityGetter
import com.xebialabs.deployit.plugin.api.udm.artifact.{Artifact, FolderArtifact}
import com.xebialabs.overthere.local.LocalConnection
import com.xebialabs.overthere.{OverthereConnection, OverthereFile}
import com.xebialabs.xldeploy.packager.io.StreamerFactory
import grizzled.slf4j.Logging

import java.io.InputStream
import java.nio.file.{Files, Paths}
import java.util.zip.ZipFile
import scala.jdk.CollectionConverters._

object TempFileMapper {
  type ZipFileEntityGetter = (String, Option[Artifact]) => Option[OverthereFile]
}

class TempFileMapper(in: InputStream) extends Logging {
  private var tempFiles: Map[String, OverthereFile] = Map.empty

  private def createTempFile(filename: String, in: InputStream)(implicit connection: OverthereConnection) = {
    val outputFile = connection.getTempFile(filename)
    val outputPath = Paths.get(outputFile.getPath)
    logger.debug(s"Writing temporary file for `$filename` at `${outputFile.toString}`")
    Files.copy(in, outputPath)
    outputFile
  }

  private def open(): (OverthereConnection, OverthereFile, ZipFile) = {
    logger.debug("Opening TempFileMapper")
    implicit val connection: OverthereConnection = LocalConnection.getLocalConnection
    val archiveFile = createTempFile("archive.zip", in)
    val zipFile = new ZipFile(Paths.get(archiveFile.getPath).toFile)
    debugListZipEntries(zipFile)
    (connection, archiveFile, zipFile)
  }

  private def debugListZipEntries(zip: ZipFile): Unit = {
    if (logger.isDebugEnabled) {
      logger.debug("Listing ZipEntries")
      zip.entries().asScala.foreach { entry =>
        logger.debug(s"ZipEntry [name: `${entry.getName}`, directory: ${entry.isDirectory}, size: ${entry.getSize} bytes], crc: [${entry.getCrc}]")
      }
    }
  }

  private def getLocalOverthereFile(path: String, artifact: Option[Artifact])
                                   (implicit connection: OverthereConnection, zipFile: ZipFile): Option[OverthereFile] = {
    if (tempFiles.contains(path)) {
      logger.debug(s"Reusing file for `$path`")
      tempFiles.get(path)
    } else {
      Option(zipFile.getEntry(path)).map(entry => {
        if (entry.isDirectory) {
          throw new AsCodeException(s"Not allowed to reference a directory: `$path`. Directories should be zipped.")
        } else {
          val in = zipFile.getInputStream(entry)
          val fileName = Paths.get(path).getFileName.toString
          val outputFileName = artifact match {
            case Some(_: FolderArtifact) =>
              if (StreamerFactory.defaultMappings().hasArchiveExtension(fileName)) fileName else s"$fileName.zip"
            case _ => fileName
          }
          val outputFile = createTempFile(outputFileName, in)
          tempFiles = tempFiles + (path -> outputFile)
          outputFile
        }
      })
    }
  }

  private def cleanup(connection: OverthereConnection, zipFile: ZipFile): Unit = {
    logger.debug("Cleaning up")
    zipFile.close()
    connection.close()
  }

  def withinLocalTempZipFile[T](callback: ZipFileEntityGetter => T): T = {
    implicit val (connection, archiveFile, zipFile) = open()
    try {
      callback(getLocalOverthereFile)
    } finally {
      cleanup(connection, zipFile)
    }
  }
}
