package com.xebialabs.deployit.ascode.service.spec

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.service.spec.{InterpreterContext, SpecInterpreter}
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.ascode.yaml.model.permission.{GlobalPermissionRelation, PermissionRelation, PermissionsSpec, RolePermissionsRelation}
import com.xebialabs.deployit.ascode.service.spec.permission.PermissionIndex
import com.xebialabs.deployit.ascode.service.spec.permission.PermissionIndex._
import com.xebialabs.deployit.ascode.yaml.model.permission.DirectoryPermissionsRelation
import com.xebialabs.deployit.core.api.InternalSecurityProxy
import com.xebialabs.deployit.core.api.dto.RolePermissions
import com.xebialabs.deployit.core.rest.api.{RoleResource, SecurityResource}
import com.xebialabs.deployit.engine.api.security.Role
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.Permission
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
@Autowired
class PermissionsSpecInterpreter(roleResource: RoleResource,
                                 internalSecurityProxy: InternalSecurityProxy,
                                 repositoryService: RepositoryService,
                                 securityResource: SecurityResource) extends SpecInterpreter {
  override def isDefinedAt(context: InterpreterContext): Boolean = context.definition.spec.isInstanceOf[PermissionsSpec]

  private def getFolderPermissions(permissions: List[PermissionRelation]): Map[String, List[RolePermissionsRelation]] =
    permissions.foldLeft(Map[String, List[RolePermissionsRelation]]()) {
      case (acc, relation: DirectoryPermissionsRelation) =>
        acc + (relation.directory -> (acc.getOrElse(relation.directory, List()) ::: relation.roles))
      case (acc, _) => acc
    }

  private def checkDirectoryPath(path: String): Unit = {
    if (!repositoryService.exists(path)) {
      throw new AsCodeException(s"Directory [$path] does not exist")
    }
    val ci = repositoryService.read[ConfigurationItem](path)
    if (!ci.isInstanceOf[Directory] && !ci.isInstanceOf[Root]) {
      throw new AsCodeException(s"[$path] is not a directory")
    }
  }

  private def validateRolePermissionsRelation(rolePermissions: RolePermissionsRelation, ciId: Option[String] = None)
                                             (implicit roles: Map[String, Role], permissionIndex: PermissionIndex): Unit = {
    if (!roles.contains(rolePermissions.role)) {
      throw new AsCodeException(s"Role [${rolePermissions.role}] does not exist")
    }
    rolePermissions.permissions.foreach(permissionIndex.checkPermission(ciId))
  }

  private def handleDirectoryPermissions(permissions: List[PermissionRelation])
                                        (implicit roles: Map[String, Role], permissionIndex: PermissionIndex): Option[ChangedIds] = {
    getFolderPermissions(permissions)
      .foldLeft(CI.ids) { case (acc, (dirPath, relations)) =>
        checkDirectoryPath(dirPath)
        val permissions = relations.map { rolePermissions =>
          validateRolePermissionsRelation(rolePermissions, Some(dirPath))
          val role = roles(rolePermissions.role)
          new RolePermissions(role, rolePermissions.permissions.asJava)
        }
        securityResource.writeRolePermissions(dirPath, permissions.asJava)
        acc.withUpdated(dirPath)
      }.toOption
  }

  private def handleGlobalPermissions(permissions: List[PermissionRelation])
                                     (implicit roles: Map[String, Role], permissionIndex: PermissionIndex): Option[ChangedIds] = {
    val storedRelations = internalSecurityProxy
      .readRolePermissions(GLOBAL_SECURITY_ALIAS, null, null, null)
      .asScala
      .map(rolePermissions => rolePermissions.getRole.getName -> rolePermissions)
      .toMap

    val globalRelations = permissions.flatMap {
      case relation: GlobalPermissionRelation => relation.global
      case _ => Nil
    }

    val newRelations = globalRelations.foldLeft(storedRelations) { case (acc, relation) =>
      validateRolePermissionsRelation(relation)
      val role = roles(relation.role)
      acc + (relation.role -> new RolePermissions(role, relation.permissions.asJava))
    }

    if (newRelations.nonEmpty) {
      internalSecurityProxy.writeRolePermissions(GLOBAL_SECURITY_ALIAS, newRelations.values.toList.asJava)
    }

    val (toUpdate, toCreate) = globalRelations.map(_.role).partition(storedRelations.keys.toSet.contains)
    PERMISSION.ids.withCreated(toCreate).withUpdated(toUpdate).toOption
  }

  override def apply(context: InterpreterContext): AsCodeResponse = {
    implicit val permissionIndex: PermissionIndex = build(Permission.getAll.asScala.toList)
    implicit val roles: Map[String, Role] = roleResource
      .list(GLOBAL_SECURITY_ALIAS, null, null, null)
      .asScala
      .map(role => role.getName -> role)
      .toMap

    val spec = context.definition.spec.asInstanceOf[PermissionsSpec]
    val changedPermissions = handleGlobalPermissions(spec.permissions)
    val changedCis = handleDirectoryPermissions(spec.permissions)

    val ids = List(changedCis, changedPermissions).flatten
    if (ids.nonEmpty) AsCodeResponse.ids(ids:_*) else AsCodeResponse()
  }
}
