package com.xebialabs.deployit.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.service.spec.InterpreterContext
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.model.{CiSpec, Spec}
import com.xebialabs.ascode.yaml.sugar.SugarConfig
import com.xebialabs.deployit.ascode.service.TempFileMapper.ZipFileEntityGetter
import com.xebialabs.deployit.ascode.yaml.parser.XLDDefinitionParser
import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider
import com.xebialabs.deployit.io.ArtifactFileUtils
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.artifact.{Artifact, SourceArtifact}
import com.xebialabs.xldeploy.packager.placeholders.SourceArtifactScanner
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.io._

object ArchiveProcessor {
  val INDEX_FILE = "index.yaml"
}

@Service
class ArchiveProcessor @Autowired()(asCodeInterpretationService: DefinitionInterpreterService,
                                    scanner: SourceArtifactScanner,
                                    parser: XLDDefinitionParser,
                                    checksumAlgorithmProvider: ChecksumAlgorithmProvider) extends Logging {

  import ArchiveProcessor.INDEX_FILE

  def process(in: InputStream, scmData: Option[SCMTraceabilityData])(implicit sugarConfig: SugarConfig): AsCodeResponse = {
    val mapper = new TempFileMapper(in)
    mapper.withinLocalTempZipFile { getEntity =>
      getEntity(INDEX_FILE, None) match {
        case Some(indexFile) =>
          val stream = indexFile.getInputStream
          val definition = parser.parse(stream)
          logger.debug(s"Read CI Spec from archive: ${definition.spec}")
          setArtifacts(definition.spec, getEntity)
          asCodeInterpretationService.interpret(InterpreterContext(definition, scmData))
        case None => AsCodeResponse.genericError(s"The applied archive does not contain an `$INDEX_FILE`")
      }
    }
  }

  private def setArtifactFile(artifact: Artifact, files: Map[String, String], getEntity: ZipFileEntityGetter): Unit = {
    files.get(artifact.getId).foreach { ref =>
      getEntity(ref, Some(artifact)) match {
        case Some(file) =>
          logger.debug(s"Setting file on `${artifact.getId}` to tempfile `${file.toString}`")
          artifact.setFile(file)
        case None => AsCodeException.throwCiFieldValidationException(
          artifact.getId, "file", s"File `$ref` referenced by `${artifact.getId}` could not be found in the archive"
        )
      }
    }
  }

  private def setArtifact(spec: CiSpec, getEntity: ZipFileEntityGetter): ConfigurationItem => Unit = {
    case artifact: SourceArtifact if artifact.getFileUri == null =>
      if (spec.files.contains(artifact.getId)) {
        setArtifactFile(artifact, spec.files, getEntity)
      } else {
        AsCodeException.throwCiFieldValidationException(
          artifact.getId, "fileUri", s"CI `${artifact.getId}` has no file or fileUri set"
        )
      }
      ArtifactFileUtils.handleArtifact(scanner, artifact, checksumAlgorithmProvider)
    case _ =>
  }

  private def setArtifacts(spec: Spec, getEntity: ZipFileEntityGetter): Unit =
    spec match {
      case ciSpec: CiSpec =>
        ciSpec
          .cis
          .filter(_.getType.instanceOf(typeOf[SourceArtifact]))
          .foreach(setArtifact(ciSpec, getEntity))
    }
}
