package com.xebialabs.deployit.cli;

import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

import java.io.File;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

public class CliOptions {
	
	static final int DEFAULT_DEPLOYIT_SERVER_PORT = 4516;
	static final String DEFAULT_DEPLOYIT_SERVER_HOST = "127.0.0.1";
	static final String EXTENSION_DIR = "ext";

	@Option(name="-secure", usage = "Use https to connect to the Deployit server", required = false)
	private boolean secured = false;

	@Option(name="-expose-proxies", required = false)
	private boolean exposeProxies = false;

	@Option(name="-host", usage="Connect to a specified host, defaults to " + DEFAULT_DEPLOYIT_SERVER_HOST + " (Optional)", required = false)
	private String host = DEFAULT_DEPLOYIT_SERVER_HOST;

	@Option(name="-port", usage="Connect to a specified port, defaults to " + DEFAULT_DEPLOYIT_SERVER_PORT + " (Optional)", required = false)
	private int port = DEFAULT_DEPLOYIT_SERVER_PORT;
	
	@Option(name="-f", usage="Execute a specified python source file (Optional)", required = false, aliases = {"-source"})
	private String sourceFileName = null;

	@Option(name="-username", usage="Connect as the specified user (Optional)", required = false)
	private String username;

	@Option(name="-password", usage="Connect with the specified password (Optional)", required = false)
	private String password;

	@Option(name="-configuration", usage="Specify the location of the configuration file (deployit.conf)", required = false)
	private String configLocation = "conf";

    @Argument(multiValued = true)
    private List<String> scriptArgs = newArrayList();
    
	private static final String DEPLOYIT_CONF_FILE_NAME = "deployit.conf";

	public void setHost(String host) {
		this.host = host;
	}

	public String getHost() {
		return host;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public int getPort() {
		return port;
	}

	public void setSourceFileName(String sourceFileName) {
		this.sourceFileName = sourceFileName;
	}

	public String getSourceFileName() {
		return sourceFileName;
	}

	public boolean interpretSourceFile() {
		return (this.sourceFileName != null);
	}

	public boolean isUsernameOnCommandline() {
		return this.username != null;
	}

	public boolean isPasswordOnCommandline() {
		return this.password != null;
	}

	public boolean isExposeProxies() {
		return exposeProxies;
	}

	public void setExposeProxies(boolean exposeProxies) {
		this.exposeProxies = exposeProxies;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(final String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(final String password) {
		this.password = password;
	}

	public String getConfigLocation() {
		return configLocation;
	}

	public void setConfigLocation(final String configLocation) {
		this.configLocation = configLocation;
	}

	public File getConfigurationFile() {
		return new File(configLocation, DEPLOYIT_CONF_FILE_NAME);
	}

	public boolean isSecured() {
		return secured;
	}

	public void setSecured(boolean secured) {
		this.secured = secured;
	}

	public String getUrl() {
		return format("http%s://%s:%d", secured ? "s" : "", host, port);
	}

    public List<String> getScriptArgs() {
        return scriptArgs;
    }

    public void setScriptArgs(List<String> scriptArgs) {
        this.scriptArgs = scriptArgs;
    }

    static CliOptions parse(String... args) {
		CliOptions options = new CliOptions();
		final CmdLineParser parser = new CmdLineParser(options);
		try {
			parser.parseArgument(args);
		} catch (CmdLineException e) {
			System.err.println(e.getMessage());
			System.err.println("java -jar deployit-cli-<version>.jar [options...]");
			parser.printUsage(System.err);
			return null;
		}
		return options;
	}
}
