package com.xebialabs.deployit.cli;

import com.xebialabs.deployit.cli.help.HelpScanner;
import jline.ConsoleReader;
import jline.Terminal;
import org.apache.commons.httpclient.auth.AuthScope;
import org.python.core.Py;
import org.python.core.PySystemState;
import org.python.jsr223.PyScriptEngine;

import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptException;
import javax.script.SimpleScriptContext;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;

public class Interpreter {
	private ScriptEngine scriptEngine;
    private CliOptions options;
    private ScriptContext scriptContext;
	private ConsoleReader consoleReader;

	public Interpreter(ConsoleReader consoleReader, ScriptEngineBuilder scriptEngineBuilder, CliOptions options) {
		this.consoleReader = consoleReader;
		this.scriptEngine = scriptEngineBuilder.build(options);
        this.options = options;
        this.scriptContext = new SimpleScriptContext();
	}

	void interpret() {
		try {
		    if (options.interpretSourceFile()) {
                scriptContext.setAttribute(ScriptEngine.FILENAME, options.getSourceFileName(), ScriptContext.ENGINE_SCOPE);
				evaluate(new FileReader(options.getSourceFileName()));
			} else {
		        try {
		            advancedRepl(consoleReader);
		        } catch (IOException e) {
		            throw new IllegalStateException("The REPL gave an exception!", e);
		        }
		    }
		} catch (FileNotFoundException e) {
		    System.err.println("File " + options.getSourceFileName() + " could not be read, reason:" + e.getMessage());
		    System.exit(1);
		} catch (ScriptException e) {
		    System.err.println("An error occurred while interpreting file " + options.getSourceFileName() + ", reason:" + e.getMessage());
		    System.exit(1);
		}

	}

	public Object evaluate(Reader reader) throws ScriptException {
		return scriptEngine.eval(reader, scriptContext);
	}

	Object evaluate(String expression) throws ScriptException {
        if (expression == null) {
            return null;
        }
		return scriptEngine.eval(expression, scriptContext);
	}

	private void advancedRepl(ConsoleReader consoleReader) throws IOException {
		for (;;) {
            final String line = consoleReader.readLine(getPrompt());
			final String trimmed = line != null ? line.trim() : line;
			if (trimmed == null || trimmed.startsWith("quit") || line.trim().startsWith("exit")) {
                exit();
				break;
			} else if (trimmed.startsWith("help")) {
				HelpScanner.printHelp();
				continue;
			}
			try {
				evaluate(line);
			} catch (ScriptException e) {
				System.err.println("Exception: " + e);
			}
		}
	}

    protected void exit() {
        System.out.println("Bye bye!");
        Terminal.resetTerminal();
    }

    String getPrompt() {
        String prompt = "deployit > ";
        String username = Cli.getAuthentication().username;
        if (username != null) {
            prompt = username + " > ";
        }
        return prompt;
    }

}
