package com.xebialabs.deployit.cli.api;

import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.api.internal.DescriptorHelper;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.QueryProxy;
import com.xebialabs.deployit.core.api.RepositoryProxy;
import com.xebialabs.deployit.core.api.dto.ArtifactAndData;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.dto.RepositoryObjectIds;
import com.xebialabs.deployit.core.api.dto.RepositoryObjects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.core.Response;

import java.util.Arrays;
import java.util.List;

@CliObject(name = "repository")
@ClassHelp(description = "Gateway to doing CRUD operations on all types of CIs")
public class RepositoryClient extends DocumentedObject {
	private final RepositoryProxy repository;
	private final QueryProxy query;
	private final DescriptorHelper descriptors;

	public RepositoryClient() {
		this.repository = null;
		this.query = null;
		this.descriptors = null;
	}

	public RepositoryClient(Proxies proxies) {
		repository = proxies.getRepository();
		query = proxies.getQuery();
		descriptors = DescriptorHelper.getInstance(proxies);
	}

	@MethodHelp(description = "Create a new CI in the repository with a specific id", parameters = {
	        @ParameterHelp(name = "id", description = "The id of the to-be-created CI"),
	        @ParameterHelp(name = "ci", description = "The CI (ConfigurationItem) that should be created in the repository") })
	public RepositoryObject create(final String id, final RepositoryObject object) {
        final Response response = repository.create(id, object);
        return checkForValidations(response);
	}

    private RepositoryObject checkForValidations(final Response response) {
        final ResponseExtractor responseExtractor = new ResponseExtractor(response);
        final RepositoryObject ci = responseExtractor.getEntity();
        if (!responseExtractor.isValidResponse() && !ci.getValidations().isEmpty()) {
            logger.error("Configuration item contained validation errors: {}", ci.getValidations());
        }
        return ci;
    }

    @MethodHelp(description = "Create a new artifact CI in the repository with a specific id", parameters = {
	        @ParameterHelp(name = "id", description = "The id of the to-be-created CI"),
	        @ParameterHelp(name = "artifact", description = "The Artifact that should be created in the repository") })
	public RepositoryObject create(final String id, final ArtifactAndData artifact) {
        final Response response = repository.create(id, artifact);
        return checkForValidations(response);
	}

	@MethodHelp(description = "Create all new CIs in the repository, commonly used after a discovery", parameters = {
	        @ParameterHelp(name = "cis", description = "The CIs (ConfigurationItems) that should be created in the repository") })
	public RepositoryObjects create(final RepositoryObjects repositoryObjects) {
		final Response response = repository.createMultiple(repositoryObjects);
		return checkAllForValidations(response);
	}

	private RepositoryObjects checkAllForValidations(Response response) {
		final ResponseExtractor responseExtractor = new ResponseExtractor(response);
		final RepositoryObjects cis = responseExtractor.getEntity();
		if (!responseExtractor.isValidResponse()) {
			for (RepositoryObject repositoryObject : cis.getObjects()) {
				if (!repositoryObject.getValidations().isEmpty()) {
					logger.error("Configuration item contained validation errors: {}", repositoryObject.getValidations());
				}
			}
		}
		return cis;
	}

	@MethodHelp(description = "Read a CI form the repository", parameters = { @ParameterHelp(name = "id", description = "The id of the CI to read") })
	public RepositoryObject read(final String id) {
		return new ResponseExtractor(repository.read(id)).getEntity();
	}

	@MethodHelp(description = "Update an existing CI in the repository with a specific id", parameters = {
	        @ParameterHelp(name = "id", description = "The id of the existing CI"),
	        @ParameterHelp(name = "ci", description = "The updated CI (ConfigurationItem) that should be stored in the repository") })
	public RepositoryObject update(final String id, final RepositoryObject object) {
        final Response response = repository.update(id, object);
        return checkForValidations(response);
	}

	@MethodHelp(description = "Update an existing artifact in the repository with a specific id", parameters = {
	        @ParameterHelp(name = "id", description = "The id of the existing artifact CI"),
	        @ParameterHelp(name = "artifact", description = "The updated artifact CI that should be stored in the repository") })
	public RepositoryObject update(final String id, final ArtifactAndData artifact) {
        final Response response = repository.update(id, artifact);
        return checkForValidations(response);
	}

	@MethodHelp(description = "Delete a CI with a specific id from the repository", parameters = { @ParameterHelp(name = "id", description = "The id of the CI") })
	public void delete(final String id) {
		new ResponseExtractor(repository.delete(id));
	}

	@MethodHelp(description = "Search for CIs of a specific type in the repository", parameters = { @ParameterHelp(name = "ciType", description = "") })
	public List<String> search(String ciType) {
		final String type = descriptors.getRegisteredType(ciType);
        if (type == null) {
            System.err.println("Configuration item type [" + type + "] not known.");
            return null;
        }
		RepositoryObjectIds ids = new ResponseExtractor(query.list(type, Boolean.FALSE, 0, -1, null, null, null)).getEntity();
		return ids.getRepositoryEntityIds();
	}

	public List<String> search(String ciType, String parent) {
		final String type = descriptors.getRegisteredType(ciType);
		RepositoryObjectIds ids = new ResponseExtractor(query.list(type, Boolean.FALSE, 0, -1, null, parent, null)).getEntity();
		return ids.getRepositoryEntityIds();
	}

	@MethodHelp(description = "Read multiple objects from the repostory in one go.", parameters = { @ParameterHelp(name = "ids", description = "The ids of the objects to read") })
	public List<RepositoryObject> read(String... ids) {
		RepositoryObjectIds repositoryObjectIds = new RepositoryObjectIds();
		repositoryObjectIds.setRepositoryEntityIds(Arrays.asList(ids));
		final RepositoryObjects objs = new ResponseExtractor(query.readMultiple(repositoryObjectIds)).getEntity();
		return objs.getObjects();
	}

    private static final Logger logger = LoggerFactory.getLogger(RepositoryClient.class);
}
