package com.xebialabs.deployit.cli.api;

import com.xebialabs.deployit.cli.Cli;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.SecurityProxy;
import com.xebialabs.deployit.core.api.ServerProxy;
import com.xebialabs.deployit.core.api.UserProxy;
import com.xebialabs.deployit.core.api.dto.RepositoryObjectIds;
import com.xebialabs.deployit.core.api.dto.User;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.core.Response;
import java.util.List;

import static com.google.common.base.Preconditions.checkArgument;

@CliObject(name = "security")
@ClassHelp(description = "Access to the security settings of Deployit.")
public class SecurityClient extends DocumentedObject {
    private final UserProxy userProxy;
	private final SecurityProxy securityProxy;
    private final ServerProxy serverProxy;

    public SecurityClient() {
        userProxy = null;
		securityProxy = null;
        serverProxy = null;
    }

    public SecurityClient(Proxies proxies) {
        userProxy = proxies.getUser();
	    securityProxy = proxies.getSecurity();
        serverProxy = proxies.getServer();
    }

    @MethodHelp(description = "Logout the currently logged in user, can only perform further actions after a login")
    public void logout() {
        Cli.getAuthentication().logout();
    }

    @MethodHelp(description = "Login a user", parameters = {
        @ParameterHelp(name = "username", description = "The username"),
        @ParameterHelp(name = "password", description = "The password")
    })
    public void login(String username, String password) {
        logger.info("Logging in as {}", username);
        Cli.getAuthentication().loginAs(username, password);
        Response info = serverProxy.getInfo();
        if (info.getStatus() > 200) {
            Cli.getAuthentication().logout();
            throw new IllegalStateException("You're not authorized with these credentials. (" + username + ")");
        }
    }

    @MethodHelp(description = "Create a user with the specified name and password", parameters = {
        @ParameterHelp(name = "username", description = "The username"),
        @ParameterHelp(name = "password", description = "The password")
    }, returns = "The created user")
    public User createUser(String username, String password) {
        return createUser(username, password, false);
    }

    public User createUser(final String username, final String password, final boolean admin) {
        User user = new User(username, admin);
        user.setPassword(password);
        return new ResponseExtractor(userProxy.create(username, user)).getEntity();
    }

    @MethodHelp(description = "Read a user so that he/she can be modified.", parameters = {
        @ParameterHelp(name = "username", description = "The username of the user to read")
    }, returns = "The read user")
    public User readUser(String username) {
        return new ResponseExtractor(userProxy.read(username)).getEntity();
    }

    @MethodHelp(description = "Modify the (password of) the user.", parameters = {
        @ParameterHelp(name = "user", description = "The updated user object.")
    })
    public void modifyUser(User user) {
        new ResponseExtractor(userProxy.modifyPassword(user.getUsername(), user));
    }

    @MethodHelp(description = "Delete a user.", parameters = {
        @ParameterHelp(name = "user", description = "the username of the user to be deleted")
    })
    public void deleteUser(String user) {
        new ResponseExtractor(userProxy.delete(user));
    }

	@MethodHelp(description = "Grant a permission to a user or group", parameters = {
		@ParameterHelp(name = "permission", description = "The permission to grant"),
		@ParameterHelp(name = "principal", description = "The user or group")
	})
	public void grant(String permission, String principal) {
		new ResponseExtractor(securityProxy.grant(permission, principal, new RepositoryObjectIds()));
	}

	@MethodHelp(description = "Revoke a permission from a user or group", parameters = {
		@ParameterHelp(name = "permission", description = "The permission to revoke"),
		@ParameterHelp(name = "principal", description = "The user or group")
	})
	public void revoke(String permission, String principal) {
		new ResponseExtractor(securityProxy.revoke(permission, principal, new RepositoryObjectIds()));
	}

	@MethodHelp(description = "Grant a permission to a user or group on a group of configuration items", parameters = {
		@ParameterHelp(name = "permission", description = "The permission to grant"),
		@ParameterHelp(name = "principal", description = "The user or group"),
		@ParameterHelp(name = "configurationItems", description = "A list of configuration items to which the permission should apply")
	})
	public void grant(String permission, String principal, List<String> configurationItems) {
		new ResponseExtractor(securityProxy.grant(permission, principal, new RepositoryObjectIds(configurationItems)));
	}

	@MethodHelp(description = "Revoke a permission from a user or group on a group of configuration items", parameters = {
		@ParameterHelp(name = "permission", description = "The permission to grant"),
		@ParameterHelp(name = "principal", description = "The user or group"),
		@ParameterHelp(name = "configurationItems", description = "A list of configuration items from which the permission should be removed")
	})
	public void revoke(String permission, String principal, List<String> configurationItems) {
		new ResponseExtractor(securityProxy.revoke(permission, principal, new RepositoryObjectIds(configurationItems)));
	}

	@MethodHelp(description = "Deny a permission to a user or group.\r\nNB: Only the 'read' permission can be denied currently.", parameters = {
		@ParameterHelp(name = "permission", description = "The permission to deny"),
		@ParameterHelp(name = "principal", description = "The user or group"),
		@ParameterHelp(name = "configurationItems", description = "A list of configuration items to which the permission should apply")
	})
	public void deny(String permission, String principal, List<String> configurationItems) {
		checkArgument(permission.equals("read"), "Only the 'read' permission can be denied currently.");
		new ResponseExtractor(securityProxy.deny(permission, principal, new RepositoryObjectIds(configurationItems)));
	}

    private static final Logger logger = LoggerFactory.getLogger(SecurityClient.class);
}
