package com.xebialabs.deployit.cli.help;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import static java.lang.System.out;

import static com.google.common.collect.Sets.newTreeSet;
import static com.google.common.collect.Lists.newArrayList;

import com.xebialabs.deployit.cli.CliObject;

public class HelpScanner {

	private static final AtomicReference<Set<Class<?>>> discoveredCliObjects = new AtomicReference<Set<Class<?>>>();

    public static void printHelp(Set<Class<?>> clazzes) {
	    discoveredCliObjects.set(clazzes);
	    printHelp();
    }

	public static void printHelp() {
    	Set<Class<?>> availableCliObjects = newTreeSet(new Comparator<Class<?>>() {
    		public int compare(Class<?> clazz, Class<?> clazzComparedTo) {
    			return clazz.getSimpleName().compareTo(clazzComparedTo.getSimpleName());
    		}
        });
        availableCliObjects.addAll(discoveredCliObjects.get());
        out.println("Deployit Objects available on the CLI:\n");
        for (Class<?> clazz : availableCliObjects) {
            final ClassHelp classHelp = clazz.getAnnotation(ClassHelp.class);
            final CliObject cliObject = clazz.getAnnotation(CliObject.class);
            if (classHelp != null) {
                out.printf("* %s: %s\n", cliObject.name(), classHelp.description());
            }
        }
        out.println("\nTo know more about a specific object, type <objectname>.help()");
        out.println("To get to know more about a specific method of an object, type <objectname>.help(\"<methodname>\")\n");
    }

    public static void printHelp(Class<?> clazz) {
        final ClassHelp classHelp = clazz.getAnnotation(ClassHelp.class);
        final CliObject cliObject = clazz.getAnnotation(CliObject.class);
        if (classHelp != null) {
            final String objectName = cliObject.name();
            out.printf("%s: %s\n\n", objectName, classHelp.description());
            printMethods(objectName, clazz);
            out.println();
        } else {
            out.println("Not found help for " + clazz);
        }
    }

    public static void printHelp(Class<?> clazz, String methodName) throws NoSuchMethodException {
        final ClassHelp classHelp = clazz.getAnnotation(ClassHelp.class);
        final CliObject cliObject = clazz.getAnnotation(CliObject.class);
        if (classHelp != null) {
            final String objectName = cliObject.name();
            for (Method method : clazz.getMethods()) {
                if (method.getName().equals(methodName)) {
                    printMethod(objectName, method);
                    printMethodDetails(method);
                    out.println();
                }
            }
        }
    }

    private static void printMethodDetails(final Method method) {
        final MethodHelp methodHelp = method.getAnnotation(MethodHelp.class);
        if (methodHelp != null) {
        	out.println();
            out.printf("Description:\n%s\n\n", methodHelp.description());
            out.print("Parameters:\n");
            if (methodHelp.parameters().length != 0) {
	            for (ParameterHelp parameterHelp : methodHelp.parameters()) {
	                out.printf("  %s: %s\n", parameterHelp.name(), parameterHelp.description());
	            }            
            } else {
            	out.print("None\n");
            }
            out.println();
            out.printf("Returns:\n%s - %s\n", method.getReturnType().getSimpleName(), methodHelp.returns());
        }
    }

    private static void printMethods(final String objectName, final Class<?> clazz) {
    	List<Method> availableMethods = newArrayList(clazz.getDeclaredMethods());
    	Collections.sort(availableMethods, new Comparator<Method>() {
    		public int compare(Method method, Method methodComparedTo) {
    			if (method.getName().equals(methodComparedTo.getName())) {
    				return 0;
    				//FIXME : if 0 then compare on number of parameters, else on length of parameters?    				
    			} else {
    				return method.getName().compareTo(methodComparedTo.getName());
    			}
        	}
        });
    	out.println("The methods available are:");
        for (Method method : availableMethods) {
            printMethod(objectName, method);
        }
    }

    private static void printMethod(final String objectName, final Method method) {
        final MethodHelp methodHelp = method.getAnnotation(MethodHelp.class);
        if (methodHelp != null) {
            String methodName = method.getName();
            StringBuilder params = new StringBuilder();
            final Class<?>[] classes = method.getParameterTypes();
            final ParameterHelp[] parameterHelps = methodHelp.parameters();
            if (classes.length != parameterHelps.length) {
                throw new IllegalArgumentException("Not all parameters are documented!");
            }

            for (int i = 0; i < classes.length; i++) {
                params.append(classes[i].getSimpleName());
                params.append(" ").append(parameterHelps[i].name());
                if (i < classes.length - 1) {
                    params.append(", ");
                }
            }
            out.printf("* %s.%s(%s) : %s\n", objectName, methodName, params, method.getReturnType().getSimpleName());
        }
    }

}
