package com.xebialabs.deployit.cli.api;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Maps.newHashMap;

import java.util.Map;
import java.util.Set;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.api.internal.DescriptorHelper;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.core.api.dto.Artifact;
import com.xebialabs.deployit.core.api.dto.ArtifactAndData;
import com.xebialabs.deployit.core.api.dto.ConfigurationItem;

@CliObject(name = "factory")
@ClassHelp(description = "Helper that can construct Configuration Items (CI) and Artifacts")
public class ObjectFactory extends DocumentedObject {
    private DescriptorHelper descriptors;

    public ObjectFactory() {
    }

    public ObjectFactory(Proxies proxies) {
        descriptors = DescriptorHelper.getInstance(proxies);
    }

    @MethodHelp(description = "Construct a CI of a specified type", parameters = {
		    @ParameterHelp(name = "id", description = "The id of the CI"),
		    @ParameterHelp(name = "ciType", description = "The type of the CI")
    })
    public ConfigurationItem configurationItem(String id, String ciType) {
        return ci(id, ciType, Maps.<String, Object>newHashMap());
    }

	@MethodHelp(description = "Construct a CI of a specified type with specified values", parameters = {
	    @ParameterHelp(name = "id", description = "The id of the CI"),
	    @ParameterHelp(name = "ciType", description = "The type of the CI"),
	    @ParameterHelp(name = "values", description = "The values of the CI in a map.")
	})
	public ConfigurationItem configurationItem(final String id, final String ciType, final Map<String, Object> values) {
		return ci(id, ciType, values);
	}

	private ConfigurationItem ci(final String id, final String ciType, final Map<String, Object> values) {
		checkArgument(isValidType(ciType), "Unknown CI type %s", ciType);

		final String type = descriptors.getRegisteredType(ciType);
		final ConfigurationItem configurationItem = new ConfigurationItem(type);
		configurationItem.setId(id);
		// Need to wrap the incoming map into a hashmap, as it could be a PyDict.
		configurationItem.setValues(Maps.<String, Object>newHashMap(values));
		return configurationItem;
	}

	@MethodHelp(description = "Construct an artifact of a specific type with associated data", parameters = {
		@ParameterHelp(name = "id", description = "The id of the artifact"),
		@ParameterHelp(name = "ciType", description = "The type of the artifact"),
		@ParameterHelp(name = "values", description = "The values for the artifact in a map."),
		@ParameterHelp(name = "data", description = "The data of the artifact that is to be uploaded")
	}, returns = "An Artifact with data that can be stored in Deployit.")
	public ArtifactAndData artifact(String id, String ciType, Map<String, Object> values, byte[] data) {
		checkArgument(isValidType(ciType), "Unknown CI type %s", ciType);

		final String type = descriptors.getRegisteredType(ciType);
		final Artifact artifact = new Artifact(type);
		artifact.setId(id);
		artifact.setValues(newHashMap(values));
		return new ArtifactAndData(artifact, data);
	}

	@MethodHelp(description = "Prints all registered configuration item types.")
	public void types() {
		final Set<String> typeSet = com.google.common.collect.Sets.newTreeSet(descriptors.types());
		System.out.printf("The registered configuration item types are:\n%s\n\n", typeSet);
	}

    private boolean isValidType(final String ciType) {
        return descriptors.isTypeRegistered(ciType);
    }

}
