package com.xebialabs.deployit.cli.api;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.sun.org.apache.xpath.internal.functions.FuncId;
import com.xebialabs.deployit.cli.Cli;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.ReferenceDataProxy;
import com.xebialabs.deployit.core.api.SecurityProxy;
import com.xebialabs.deployit.core.api.ServerProxy;
import com.xebialabs.deployit.core.api.UserProxy;
import com.xebialabs.deployit.core.api.dto.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.core.Response;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;

@CliObject(name = "security")
@ClassHelp(description = "Access to the security settings of Deployit.")
public class SecurityClient extends DocumentedObject {
	private final UserProxy userProxy;
	private final SecurityProxy securityProxy;
    private final ServerProxy serverProxy;
	private final ReferenceDataProxy referenceDataProxy;

	public SecurityClient() {
        userProxy = null;
		securityProxy = null;
        serverProxy = null;
		referenceDataProxy = null;
    }

    public SecurityClient(Proxies proxies) {
        userProxy = proxies.getUser();
	    securityProxy = proxies.getSecurity();
        serverProxy = proxies.getServer();
	    referenceDataProxy = proxies.getReferenceData();
    }

    @MethodHelp(description = "Logout the currently logged in user, can only perform further actions after a login")
    public void logout() {    	
    	Cli.getAuthentication().logout();
    }

    @MethodHelp(description = "Login a user", parameters = {
        @ParameterHelp(name = "username", description = "The username"),
        @ParameterHelp(name = "password", description = "The password")
    })
    public void login(String username, String password) {
        logger.info("Logging in as {}", username);
        Cli.getAuthentication().loginAs(username, password);
        Response info = serverProxy.getInfo();
        if (info.getStatus() > 200) {
            Cli.getAuthentication().logout();
            throw new IllegalStateException("You're not authorized with these credentials. (" + username + ")");
        }
    }

    @MethodHelp(description = "Create a user with the specified name and password", parameters = {
        @ParameterHelp(name = "username", description = "The username"),
        @ParameterHelp(name = "password", description = "The password")
    }, returns = "The created user")
    public User createUser(String username, String password) {
        return createUser(username, password, false);
    }

    public User createUser(final String username, final String password, final boolean admin) {
        User user = new User(username, admin);
        user.setPassword(password);
        return new ResponseExtractor(userProxy.create(username, user)).getEntity();
    }

    @MethodHelp(description = "Read a user so that he/she can be modified.", parameters = {
        @ParameterHelp(name = "username", description = "The username of the user to read")
    }, returns = "The read user")
    public User readUser(String username) {
        return new ResponseExtractor(userProxy.read(username)).getEntity();
    }

    @MethodHelp(description = "Modify the (password of) the user.", parameters = {
        @ParameterHelp(name = "user", description = "The updated user object.")
    })
    public void modifyUser(User user) {
        new ResponseExtractor(userProxy.modifyPassword(user.getUsername(), user));
    }

    @MethodHelp(description = "Delete a user.", parameters = {
        @ParameterHelp(name = "username", description = "the username of the user to be deleted")
    })
    public void deleteUser(String username) {
        new ResponseExtractor(userProxy.delete(username));
    }

	@MethodHelp(description = "Grant a permission to a role", parameters = {
		@ParameterHelp(name = "permission", description = "The permission to grant"),
		@ParameterHelp(name = "roleName", description = "The role")
	})
	public void grant(String permission, String roleName) {
		doGrant(permission, roleName, "global");
	}

	@MethodHelp(description = "Grant a permission to a role on a group of configuration items", parameters = {
			@ParameterHelp(name = "permission", description = "The permission to grant"),
			@ParameterHelp(name = "roleName", description = "The role"),
			@ParameterHelp(name = "configurationItems", description = "A list of configuration items to which the permission should apply")
	})
	public void grant(String permission, String roleName, List<String> configurationItems) {
		List<String> ids = removeTrailingSlashes(configurationItems);
		for (String id : ids) {
			doGrant(permission, roleName, id);
		}
	}

	@MethodHelp(description = "Revoke a permission from a role", parameters = {
		@ParameterHelp(name = "permission", description = "The permission to revoke"),
		@ParameterHelp(name = "roleName", description = "The role")
	})
	public void revoke(String permission, String roleName) {
		doRevoke(permission, roleName, "global");
	}

	@MethodHelp(description = "Revoke a permission from a role on a group of configuration items", parameters = {
			@ParameterHelp(name = "permission", description = "The permission to grant"),
			@ParameterHelp(name = "roleName", description = "The role"),
			@ParameterHelp(name = "configurationItems", description = "A list of configuration items from which the permission should be removed")
	})
	public void revoke(String permission, String roleName, List<String> configurationItems) {
		List<String> ids = removeTrailingSlashes(configurationItems);
		for (String id : ids) {
			doRevoke(permission, roleName, id);
		}
	}

	private void doGrant(String permission, String role, String id) {
		RolePermissions permissions = new ResponseExtractor(securityProxy.getGrantedPermissions(id)).getEntity();
		permissions.get(role).grant(permission);
		new ResponseExtractor(securityProxy.writeGrantedPermissions(id, permissions));
	}

	private void doRevoke(String permission, String role, String id) {
		RolePermissions permissions = new ResponseExtractor(securityProxy.getGrantedPermissions(id)).getEntity();
		permissions.get(role).revoke(permission);
		new ResponseExtractor(securityProxy.writeGrantedPermissions(id, permissions));
	}


	@MethodHelp(description = "Check whether a permission is granted to the logged in user on an id.", parameters = {
			@ParameterHelp(name = "permission", description = "The permission to check"),
			@ParameterHelp(name = "principal", description = "The user or group whose permissions are to be retrieved")},
			returns = ("A permissions object containing the user's or group's granted permissions"))
	public boolean hasPermission(String permission, String id) {
		try {
			new ResponseExtractor(securityProxy.hasPermission(permission, id));
			return true;
		} catch (Exception e) {
			return false;
		}
	}
	
	@MethodHelp(description = "Get permissions granted to a user or group", parameters = {
		@ParameterHelp(name = "principal", description = "The user or group whose permissions are to be retrieved")},
		returns = ("A permissions object containing the user's or group's granted permissions"))
	public PrincipalPermissions getPermissions(final String principal) {
		return new ResponseExtractor(securityProxy.retrievePermissions(principal)).getEntity();
	}

	@MethodHelp(description = "Get permissions granted to the currently logged in user")
	public PrincipalPermissions getPermissions() {
		return getPermissions(Cli.getAuthentication().getUserName());
	}

	@MethodHelp(description = "List all the permissions available in Deployit")
	public void printPermissions() {
		Permissions permissions = new ResponseExtractor(referenceDataProxy.permissionList()).getEntity();
		System.out.println("Available permissions are:");
		for (Permission s : permissions.getPermissions()) {
			System.out.println(s.getPermissionName());
		}
	}

    @MethodHelp(description = "Create or update a role with assigned principals.", parameters = {
        @ParameterHelp(name = "roleName", description = "The role name"),
        @ParameterHelp(name = "principals", description = "The assigned principals")
    })
	public void assignRole(String roleName, List<String> principals) {
		RoleAssignments assignments = new ResponseExtractor(securityProxy.getRoleAssignments()).getEntity();
		assignments.assignRole(roleName, principals);
		new ResponseExtractor(securityProxy.writeRoleAssignments(assignments));
	}

    @MethodHelp(description = "Remove a role.", parameters = {
            @ParameterHelp(name = "roleName", description = "The role name")
    })
	public void removeRole(String roleName) {
		RoleAssignments assignments = new ResponseExtractor(securityProxy.getRoleAssignments()).getEntity();
		assignments.removeRole(roleName);
		new ResponseExtractor(securityProxy.writeRoleAssignments(assignments));
	}

    @MethodHelp(description = "Get all existing roles in Deployit.", returns = "The list of role names")
    public List<String> getRoleNames() {
        Roles roles = new ResponseExtractor(securityProxy.getRoles()).getEntity();
        return newArrayList(Lists.transform(roles.getRoles(), new Function<Role, String>() {
            public String apply(Role input) {
                return input.getName();
            }
        }));
    }
	
	public void renameRole(String oldName, String newName) {
		RoleAssignments roles = new ResponseExtractor(securityProxy.getRoleAssignments()).getEntity();
		for (RoleAssignment role : roles.getAssignments()) {
			if (role.getRole().getName().equals(oldName)) {
				role.getRole().setName(newName);
				break;
			}
		}
		new ResponseExtractor(securityProxy.writeRoleAssignments(roles));
	}

    @MethodHelp(description = "Get the principals assigned to a specific role.", parameters = {
            @ParameterHelp(name = "roleName", description = "The role name")
    }, returns = "A list of principals assigned to the role")
    public List<String> getRoleAssignments(String roleName) {
        RoleAssignments roleAssignments = new ResponseExtractor(securityProxy.getRoleAssignments()).getEntity();
        List<RoleAssignment> assignments = roleAssignments.getAssignments();
        for (RoleAssignment assignment : assignments) {
            if (assignment.getRole().getName().equals(roleName)) {
                return assignment.getPrincipals();
            }
        }
        return newArrayList();
    }
	
	private List<String> removeTrailingSlashes(final List<String> repositoryEntityIds) {
		List<String> cleanRepositoryEntityIds = Lists.<String>newArrayList();
		for (String repositoryEntityId : repositoryEntityIds) {
			cleanRepositoryEntityIds.add(repositoryEntityId.replaceFirst("/*$", ""));
		}
		return cleanRepositoryEntityIds;
	}

    private static final Logger logger = LoggerFactory.getLogger(SecurityClient.class);
}
